#!/bin/bash
# NZBGet Status Checker
# Usage: ./check_nzbget.sh [count|status|speed|queue]
# Required env vars: NZBGET_USER, NZBGET_PASS, NZBGET_HOST

HOST="${NZBGET_HOST:-}"
USER="${NZBGET_USER:-}"
PASS="${NZBGET_PASS:-}"

# Check if credentials are set
if [ -z "$USER" ] || [ -z "$PASS" ] || [ -z "$HOST" ]; then
    echo "Error: NZBGET_USER and NZBGET_PASS and NZBGET_HOST environment variables must be set"
    exit 1
fi

URL="http://${USER}:${PASS}@${HOST}/jsonrpc"

# Helper function to make NZBGet API calls
nzbget_api() {
    local method="$1"
    curl -s -X POST -H "Content-Type: application/json" \
        -d "{\"method\": \"${method}\", \"id\": 1}" \
        "$URL" 2>/dev/null
}

# Count mode - count ALL items in queue (any status)
if [ "$1" = "count" ]; then
    QUEUE=$(nzbget_api "listgroups")
    if [ -z "$QUEUE" ] || [ "$QUEUE" = "null" ]; then
        echo "0"
        exit 0
    fi
    TOTAL_COUNT=$(echo "$QUEUE" | jq '[.result[]?] | length' 2>/dev/null)
    echo "${TOTAL_COUNT:-0}"
    exit 0
fi

# Speed mode - current download speed only
if [ "$1" = "speed" ]; then
    STATUS=$(nzbget_api "status")
    if [ -z "$STATUS" ] || [ "$STATUS" = "null" ]; then
        echo "NZBGet unreachable"
        exit 1
    fi
    SPEED=$(echo "$STATUS" | jq -r '.result.DownloadRate // 0' 2>/dev/null)
    # Convert bytes/sec to MB/s
    SPEED_MB=$(echo "scale=1; $SPEED / 1048576" | bc 2>/dev/null || echo "0")
    echo "${SPEED_MB} MB/s"
    exit 0
fi

# Queue mode - list first 10 queue items only
if [ "$1" = "queue" ]; then
    QUEUE=$(nzbget_api "listgroups")
    if [ -z "$QUEUE" ] || [ "$QUEUE" = "null" ]; then
        echo "NZBGet unreachable"
        exit 1
    fi
    
    TOTAL_QUEUE=$(echo "$QUEUE" | jq '[.result[]?] | length' 2>/dev/null || echo "0")
    
    if [ "$TOTAL_QUEUE" -gt 10 ]; then
        echo "Next 10 of ${TOTAL_QUEUE} items:"
        echo ""
    fi
    
    echo "$QUEUE" | jq -r '.result[:10][]? | 
        "\(.NZBName) | \(.Status) | \(.SizeMB // 0 | tonumber / 1024 | floor) GB | \(.Progress // 0 | floor)%"' 2>/dev/null
    exit 0
fi

# Full status mode (default)
QUEUE=$(nzbget_api "listgroups")
STATUS=$(nzbget_api "status")

if [ -z "$QUEUE" ] || [ "$QUEUE" = "null" ]; then
    echo "❌ NZBGet unreachable at ${HOST}"
    exit 1
fi

if [ -z "$STATUS" ] || [ "$STATUS" = "null" ]; then
    echo "❌ NZBGet status unavailable"
    exit 1
fi

# Parse data
TOTAL_COUNT=$(echo "$QUEUE" | jq '[.result[]?] | length' 2>/dev/null || echo "0")
DOWNLOADING_COUNT=$(echo "$QUEUE" | jq '[.result[]? | select(.Status == "DOWNLOADING")] | length' 2>/dev/null || echo "0")
SPEED=$(echo "$STATUS" | jq -r '.result.DownloadRate // 0' 2>/dev/null)
SPEED_MB=$(echo "scale=1; $SPEED / 1048576" | bc 2>/dev/null || echo "0")
STATE=$(echo "$STATUS" | jq -r '.result.DownloadStatus // "Unknown"' 2>/dev/null)
SIZE_LEFT=$(echo "$STATUS" | jq -r '.result.RemainingSizeMB // 0' 2>/dev/null)
SIZE_LEFT_GB=$(echo "scale=1; $SIZE_LEFT / 1024" | bc 2>/dev/null || echo "0")

# Output summary
echo "📥 NZBGet Status: ${STATE}"
echo ""
echo "Queue Size: ${TOTAL_COUNT} items"
echo "Speed: ${SPEED_MB} MB/s"
echo "Remaining: ${SIZE_LEFT_GB} GB"

# List next 10 items in queue (all statuses)
if [ "$TOTAL_COUNT" -gt 0 ]; then
    echo ""
    if [ "$TOTAL_COUNT" -gt 10 ]; then
        echo "Next 10 items (of ${TOTAL_COUNT} total in queue):"
    else
        echo "Queue (${TOTAL_COUNT} items):"
    fi
    echo "$QUEUE" | jq -r '.result[:10][]? | 
        "  • [\(.Status)] \(.NZBName) - \(.Progress // 0 | floor)%"' 2>/dev/null
    
    if [ "$TOTAL_COUNT" -gt 10 ]; then
        echo "  ... and $((TOTAL_COUNT - 10)) more"
    fi
fi

exit 0
