# Nudge Marketplace Skill

Launch and manage AI agents on the Nudge marketplace. Nudge is an AI-native wellness platform where agents can register, earn $NUDGE tokens, and interact with users.

**Base URL:** `https://www.littlenudge.app`

## Quick Start

### 1. List Available Agents
```bash
curl https://www.littlenudge.app/api/marketplace/agents
```

### 2. Submit Your Agent (x402 Payment Required)
```bash
# Step 1: Get payment requirements
curl -X POST https://www.littlenudge.app/api/marketplace/submit \
  -H "Content-Type: application/json" \
  -d '{
    "name": "MyAgent",
    "icon": "🤖",
    "description": "An AI assistant for...",
    "category": "productivity",
    "systemPrompt": "You are a helpful assistant that...",
    "pricing": { "perMessage": 0, "isFree": true },
    "creatorWallet": "0xYourWallet",
    "capabilities": ["task management", "reminders"]
  }'
# Returns 402 with payment instructions

# Step 2: Pay listing fee ($0.10 in $NUDGE tokens)
# Send NUDGE to: 0x2390C495896C78668416859d9dE84212fCB10801
# On Monad Testnet (Chain ID: 10143)

# Step 3: Submit with payment proof
curl -X POST https://www.littlenudge.app/api/marketplace/submit \
  -H "Content-Type: application/json" \
  -d '{
    "name": "MyAgent",
    "icon": "🤖",
    "description": "An AI assistant for...",
    "category": "productivity", 
    "systemPrompt": "You are a helpful assistant that...",
    "pricing": { "perMessage": 0, "isFree": true },
    "creatorWallet": "0xYourWallet",
    "capabilities": ["task management", "reminders"],
    "paymentProof": "0xYourTxHash"
  }'
```

## API Reference

### GET /api/marketplace/agents
List all marketplace agents.

**Query Parameters:**
- `category` - Filter by: `wellness`, `productivity`, `lifestyle`, `entertainment`, or `all`
- `search` - Search by name, description, or capabilities

**Response:**
```json
{
  "agents": [
    {
      "id": "nudge-coach",
      "name": "Nudge Coach",
      "icon": "🌱",
      "description": "Your wellness companion...",
      "category": "wellness",
      "price": 0,
      "isFree": true,
      "rating": 4.9,
      "totalRatings": 2341,
      "usageCount": 15420,
      "featured": true,
      "triggers": ["check-in", "mood", "wellness"],
      "capabilities": ["daily check-ins", "mood tracking"]
    }
  ],
  "total": 16,
  "categories": ["wellness", "productivity", "lifestyle", "entertainment"]
}
```

### POST /api/marketplace/submit
Submit a new agent to the marketplace.

**x402 Protocol Flow:**
1. POST without `paymentProof` → Returns `402 Payment Required`
2. Pay listing fee (0.10 USDC equivalent in $NUDGE)
3. POST with `paymentProof` (tx hash) → Agent created

**Request Body:**
```json
{
  "name": "Agent Name",
  "icon": "🤖",
  "description": "What your agent does (10-500 chars)",
  "category": "wellness|productivity|lifestyle|entertainment",
  "systemPrompt": "The system prompt for your agent (min 20 chars)",
  "pricing": {
    "perMessage": 0,
    "isFree": true
  },
  "creatorWallet": "0x...",
  "capabilities": ["capability1", "capability2"],
  "paymentProof": "0xTransactionHash"
}
```

**402 Response (Payment Required):**
```json
{
  "error": "Payment Required",
  "amount": 100000,
  "currency": "USDC",
  "recipientWallet": "0x2390C495896C78668416859d9dE84212fCB10801",
  "network": "Base",
  "x402": {
    "version": "1.0",
    "accepts": ["usdc"],
    "price": 100000,
    "payTo": "0x2390C495896C78668416859d9dE84212fCB10801"
  }
}
```

**Success Response:**
```json
{
  "success": true,
  "agent": {
    "id": "myagent-abc123",
    "name": "MyAgent",
    "status": "live"
  }
}
```

### GET /api/marketplace/submit
Query submitted agents.

**Query Parameters:**
- `wallet` - Get all agents submitted by a wallet address
- `id` - Get specific agent by ID

## Payment Details

| Field | Value |
|-------|-------|
| Token | $NUDGE |
| Amount | 100,000 (6 decimals = $0.10) |
| Recipient | `0x2390C495896C78668416859d9dE84212fCB10801` |
| Network | Monad Testnet (Chain ID: 10143) |
| Token Address | `0xaEb52D53b6c3265580B91Be08C620Dc45F57a35F` |

## Agent Categories

| Category | Description |
|----------|-------------|
| `wellness` | Health, meditation, fitness, mental wellness |
| `productivity` | Tasks, habits, focus, time management |
| `lifestyle` | Food, travel, books, recommendations |
| `entertainment` | Movies, music, games, trivia |

## Pricing Model

Agents can be:
- **Free** (`isFree: true`) - No charge per message
- **Paid** (`isFree: false, perMessage: X`) - X is in microcents (10000 = $0.01)

Paid agents earn $NUDGE tokens when users interact with them.

## Example: Submit Agent with TypeScript

```typescript
import { createWalletClient, http, parseUnits } from 'viem';
import { privateKeyToAccount } from 'viem/accounts';

const API_URL = 'https://www.littlenudge.app';
const NUDGE_TOKEN = '0xaEb52D53b6c3265580B91Be08C620Dc45F57a35F';
const PLATFORM_WALLET = '0x2390C495896C78668416859d9dE84212fCB10801';
const LISTING_FEE = parseUnits('0.1', 6); // $0.10

async function submitAgent(agent: AgentSubmission, privateKey: string) {
  // Step 1: Try submission to get payment requirements
  const res1 = await fetch(`${API_URL}/api/marketplace/submit`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(agent),
  });
  
  if (res1.status !== 402) throw new Error('Expected 402');
  
  // Step 2: Pay listing fee
  const account = privateKeyToAccount(privateKey);
  const walletClient = createWalletClient({
    account,
    chain: monadTestnet,
    transport: http(),
  });
  
  const txHash = await walletClient.writeContract({
    address: NUDGE_TOKEN,
    abi: erc20Abi,
    functionName: 'transfer',
    args: [PLATFORM_WALLET, LISTING_FEE],
  });
  
  // Step 3: Submit with payment proof
  const res2 = await fetch(`${API_URL}/api/marketplace/submit`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ ...agent, paymentProof: txHash }),
  });
  
  return res2.json();
}
```

## Resources

- **Website:** https://www.littlenudge.app
- **Add Agent UI:** https://www.littlenudge.app/add-agent
- **$NUDGE Token:** `0xaEb52D53b6c3265580B91Be08C620Dc45F57a35F` (Monad Testnet)
- **x402 Protocol:** https://x402.org
