#!/usr/bin/env node
/**
 * notionctl.mjs
 *
 * A JSON-first CLI for common Notion operations tuned for agent use:
 * - Always uses Notion-Version 2025-09-03
 * - Retries on 429 with Retry-After, and on transient 5xx
 * - Chunks block appends to avoid API limits
 * - Exports pages to Markdown (best-effort) and imports simple Markdown to blocks
 *
 * No external deps (Node 18+ fetch). Tested with Node 22.
 */

import fs from "node:fs";
import os from "node:os";
import path from "node:path";
import process from "node:process";

const NOTION_API_BASE = "https://api.notion.com";
const NOTION_VERSION = "2025-09-03";

// Request limits say 3 req/s average; be conservative to reduce 429s.
const MIN_INTERVAL_MS = 350;

const MAX_RETRIES = 6;
const MAX_BLOCKS_PER_APPEND = 100; // matches "Any array of all block types ... 100 elements"
const DEFAULT_PAGE_SIZE = 100;

let nextAllowedAt = 0;

/** ---------- small utilities ---------- **/

function sleep(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

function nowMs() {
  return Date.now();
}

function isObject(x) {
  return x !== null && typeof x === "object" && !Array.isArray(x);
}

function compactJson(x) {
  return JSON.stringify(x);
}

function prettyJson(x) {
  return JSON.stringify(x, null, 2);
}

function print(obj, { compact = false } = {}) {
  process.stdout.write((compact ? compactJson(obj) : prettyJson(obj)) + "\n");
}

function fail(message, { code = 1, details } = {}) {
  const err = { ok: false, error: message };
  if (details !== undefined) err.details = details;
  print(err, { compact: false });
  process.exit(code);
}

function usage() {
  const msg = `
notionctl — Notion API CLI for agents (JSON-first)

Usage:
  notionctl whoami
  notionctl search --query "text" [--type page|data_source|all] [--limit 20]
  notionctl get-page --page "<id-or-url>"
  notionctl export-md --page "<id-or-url>" [--stdout-md]
  notionctl create-md --title "Title" (--parent-page "<id-or-url>" | --parent-data-source "<id-or-url>") (--md "..." | --md-file path | --md-stdin)
                [--set "Prop=Value" ...]
                [--template none|default | --template-id "<id-or-url>"]
                [--position page_start|page_end | --after-block "<id-or-url>"]   (page parents only)
  notionctl append-md --page "<id-or-url>" (--md "..." | --md-file path | --md-stdin)
  notionctl move --page "<id-or-url>" (--to-page "<id-or-url>" | --to-data-source "<id-or-url>")
  notionctl list-child-pages --page "<id-or-url>"
  notionctl triage --inbox-page "<id-or-url>" --rules "<json-file>" [--limit 50] [--apply]

Common flags:
  --compact        output single-line JSON
  --help           show help

Environment:
  NOTION_API_KEY (preferred)
  NOTION_TOKEN / NOTION_API_TOKEN (fallbacks)

Local fallback:
  ~/.config/notion/api_key

Notes:
  - All requests use Notion-Version: ${NOTION_VERSION}.
  - export-md is best-effort: it covers common blocks used for notes.
`.trim();
  process.stdout.write(msg + "\n");
}

function parseArgs(argv) {
  // Minimal flag parser: supports --k v, --k=v, repeated flags, and positionals in _.
  const out = { _: [] };
  for (let i = 0; i < argv.length; i++) {
    const tok = argv[i];
    if (!tok.startsWith("--")) {
      out._.push(tok);
      continue;
    }
    const eq = tok.indexOf("=");
    if (eq !== -1) {
      const k = tok.slice(2, eq);
      const v = tok.slice(eq + 1);
      pushFlag(out, k, v);
      continue;
    }
    const k = tok.slice(2);
    const next = argv[i + 1];
    if (next === undefined || next.startsWith("--")) {
      pushFlag(out, k, true);
    } else {
      pushFlag(out, k, next);
      i++;
    }
  }
  return out;
}

function pushFlag(obj, key, value) {
  if (obj[key] === undefined) {
    obj[key] = value;
  } else if (Array.isArray(obj[key])) {
    obj[key].push(value);
  } else {
    obj[key] = [obj[key], value];
  }
}

function resolveHome(p) {
  if (!p) return p;
  if (p.startsWith("~" + path.sep) || p === "~") {
    return path.join(os.homedir(), p.slice(1));
  }
  return p;
}

/** ---------- Notion IDs ---------- **/

function toDashedUuid(hex32) {
  const s = hex32.toLowerCase();
  return `${s.slice(0, 8)}-${s.slice(8, 12)}-${s.slice(12, 16)}-${s.slice(16, 20)}-${s.slice(20)}`;
}

function normaliseId(idOrUrl) {
  if (!idOrUrl || typeof idOrUrl !== "string") throw new Error("Missing ID/URL");
  const s = idOrUrl.trim();

  // Prefer dashed UUID if present.
  const dashed = s.match(/[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}/);
  if (dashed) return dashed[0].toLowerCase();

  // Else a 32-hex ID embedded in URLs.
  const hex32 = s.match(/[0-9a-fA-F]{32}/);
  if (hex32) return toDashedUuid(hex32[0]);

  throw new Error(`Could not extract a Notion UUID from: ${idOrUrl}`);
}

/** ---------- auth ---------- **/

function readTokenFromEnv() {
  return (
    process.env.NOTION_API_KEY ||
    process.env.NOTION_TOKEN ||
    process.env.NOTION_API_TOKEN ||
    ""
  ).trim();
}

function readTokenFromFile() {
  const p = resolveHome(path.join("~", ".config", "notion", "api_key"));
  try {
    if (!fs.existsSync(p)) return "";
    return fs.readFileSync(p, "utf8").trim();
  } catch {
    return "";
  }
}

function getToken() {
  const env = readTokenFromEnv();
  if (env) return env;
  const file = readTokenFromFile();
  if (file) return file;
  throw new Error(
    "Missing Notion token. Set NOTION_API_KEY (recommended) or create ~/.config/notion/api_key"
  );
}

/** ---------- HTTP ---------- **/

function buildHeaders(token, { hasBody = false } = {}) {
  const h = {
    Authorization: `Bearer ${token}`,
    "Notion-Version": NOTION_VERSION,
    Accept: "application/json",
  };
  if (hasBody) h["Content-Type"] = "application/json";
  return h;
}

async function throttle() {
  const t = nowMs();
  if (t < nextAllowedAt) await sleep(nextAllowedAt - t);
  nextAllowedAt = nowMs() + MIN_INTERVAL_MS;
}

async function notionRequest({ method, path, query, body }) {
  const token = getToken();
  const url = new URL(NOTION_API_BASE + path);
  if (query) {
    for (const [k, v] of Object.entries(query)) {
      if (v === undefined || v === null) continue;
      url.searchParams.set(k, String(v));
    }
  }

  const hasBody = body !== undefined && body !== null;
  const init = {
    method,
    headers: buildHeaders(token, { hasBody }),
  };
  if (hasBody) init.body = JSON.stringify(body);

  for (let attempt = 0; attempt <= MAX_RETRIES; attempt++) {
    await throttle();
    const res = await fetch(url, init);

    // Success
    if (res.ok) {
      const text = await res.text();
      if (!text) return null;
      try {
        return JSON.parse(text);
      } catch {
        return text;
      }
    }

    // Parse error response body (best-effort)
    let errBody = null;
    try {
      errBody = await res.json();
    } catch {
      try {
        errBody = await res.text();
      } catch {
        errBody = null;
      }
    }

    // Retry logic
    const status = res.status;

    if (status === 429 && attempt < MAX_RETRIES) {
      const retryAfter = res.headers.get("retry-after");
      const waitSec = retryAfter ? Number.parseFloat(retryAfter) : NaN;
      const waitMs = Number.isFinite(waitSec) ? Math.max(0, waitSec) * 1000 : 1000 * Math.pow(2, attempt);
      await sleep(waitMs);
      continue;
    }

    if ((status === 500 || status === 502 || status === 503 || status === 504) && attempt < MAX_RETRIES) {
      const waitMs = 500 * Math.pow(2, attempt);
      await sleep(waitMs);
      continue;
    }

    // Non-retryable or retries exhausted
    const msg = isObject(errBody) && typeof errBody.message === "string"
      ? errBody.message
      : `HTTP ${status}`;
    const code = isObject(errBody) ? errBody.code : undefined;
    const error = { status, code, message: msg, body: errBody };
    throw new Error(prettyJson(error));
  }

  throw new Error("Request failed after retries");
}

/** ---------- Notion helpers ---------- **/

function getPageTitle(page) {
  // For database rows, title property name is not always "title".
  if (!page || !isObject(page)) return null;
  const props = page.properties;
  if (!isObject(props)) return null;

  for (const [name, prop] of Object.entries(props)) {
    if (isObject(prop) && prop.type === "title" && Array.isArray(prop.title)) {
      return prop.title.map((t) => t.plain_text ?? "").join("");
    }
  }

  // Fallback commonly present in many responses.
  const direct = props.title;
  if (isObject(direct) && direct.type === "title" && Array.isArray(direct.title)) {
    return direct.title.map((t) => t.plain_text ?? "").join("");
  }

  return null;
}

function getRichTextArray(obj, typeKey) {
  if (!isObject(obj)) return [];
  const inner = obj[typeKey];
  if (!isObject(inner)) return [];
  const rt = inner.rich_text;
  if (!Array.isArray(rt)) return [];
  return rt;
}

function escapeMd(text) {
  // Minimal escape to avoid accidental headings/lists.
  return String(text).replace(/\\/g, "\\\\");
}

function applyAnnotations(text, ann) {
  if (!ann || !isObject(ann)) return text;
  let out = text;
  // Order matters (code first avoids double-wrapping).
  if (ann.code) out = "`" + out.replace(/`/g, "\\`") + "`";
  if (ann.bold) out = "**" + out + "**";
  if (ann.italic) out = "*" + out + "*";
  if (ann.strikethrough) out = "~~" + out + "~~";
  if (ann.underline) out = "<u>" + out + "</u>";
  return out;
}

function richTextToMarkdown(richText) {
  if (!Array.isArray(richText) || richText.length === 0) return "";
  return richText
    .map((rt) => {
      if (!isObject(rt)) return "";
      const ann = rt.annotations;
      const plain = rt.plain_text ?? "";
      let txt = escapeMd(plain);

      // Prefer explicit link, then href.
      const url = (rt.type === "text" && rt.text && rt.text.link && rt.text.link.url) ? rt.text.link.url : rt.href;
      txt = applyAnnotations(txt, ann);
      if (url) txt = `[${txt}](${url})`;

      return txt;
    })
    .join("");
}

function textToRichText(text) {
  // Notion has a 2000-char limit for text.content; split conservatively.
  const s = String(text ?? "");
  const out = [];
  for (let i = 0; i < s.length; i += 2000) {
    out.push({
      type: "text",
      text: { content: s.slice(i, i + 2000) },
    });
  }
  return out;
}

/** ---------- Blocks: fetch + markdown export ---------- **/

async function listBlockChildren(blockId) {
  const id = normaliseId(blockId);
  const results = [];
  let cursor = undefined;
  while (true) {
    const q = { page_size: DEFAULT_PAGE_SIZE };
    if (cursor) q.start_cursor = cursor;

    const res = await notionRequest({
      method: "GET",
      path: `/v1/blocks/${id}/children`,
      query: q,
    });

    if (res && Array.isArray(res.results)) results.push(...res.results);
    if (!res || !res.has_more) break;
    cursor = res.next_cursor;
    if (!cursor) break;
  }
  return results;
}

async function getBlockTree(blockId, { maxDepth = 50, depth = 0 } = {}) {
  const blocks = await listBlockChildren(blockId);
  if (depth >= maxDepth) return blocks;

  for (const b of blocks) {
    if (b && b.has_children) {
      b.children = await getBlockTree(b.id, { maxDepth, depth: depth + 1 });
    }
  }
  return blocks;
}

function blockToMarkdown(block, indent = 0) {
  const pad = " ".repeat(indent);
  const type = block.type;
  const obj = block[type] || {};

  // Common rich text-based blocks
  const rt = isObject(obj) && Array.isArray(obj.rich_text) ? obj.rich_text : [];
  const txt = richTextToMarkdown(rt);

  const children = Array.isArray(block.children) ? block.children : [];

  const renderChildren = (childIndent) => {
    if (!children.length) return "";
    const parts = children.map((c) => blockToMarkdown(c, childIndent)).filter(Boolean);
    const joined = parts.join("\n");
    return joined ? "\n" + joined : "";
  };

  switch (type) {
    case "paragraph": {
      const line = txt ? pad + txt : "";
      return (line + renderChildren(indent)).trimEnd();
    }
    case "heading_1":
      return (pad + "# " + txt).trimEnd() + renderChildren(indent);
    case "heading_2":
      return (pad + "## " + txt).trimEnd() + renderChildren(indent);
    case "heading_3":
      return (pad + "### " + txt).trimEnd() + renderChildren(indent);
    case "bulleted_list_item": {
      const line = pad + "- " + txt;
      return (line + renderChildren(indent + 2)).trimEnd();
    }
    case "numbered_list_item": {
      const line = pad + "1. " + txt;
      return (line + renderChildren(indent + 3)).trimEnd();
    }
    case "to_do": {
      const checked = !!obj.checked;
      const line = pad + `- [${checked ? "x" : " "}] ` + txt;
      return (line + renderChildren(indent + 2)).trimEnd();
    }
    case "quote": {
      const line = pad + "> " + txt;
      return (line + renderChildren(indent)).trimEnd();
    }
    case "code": {
      const lang = obj.language || "";
      const codeText = (Array.isArray(obj.rich_text) ? obj.rich_text.map((t) => t.plain_text ?? "").join("") : "");
      return `${pad}\`\`\`${lang}\n${codeText}\n${pad}\`\`\``;
    }
    case "divider":
      return pad + "---";
    case "callout": {
      const icon = block.icon && block.icon.type === "emoji" ? block.icon.emoji + " " : "";
      const line = pad + "> " + icon + txt;
      return (line + renderChildren(indent)).trimEnd();
    }
    case "toggle": {
      // Notion toggles map best to HTML details in Markdown.
      const summary = txt || "Details";
      const inner = children.map((c) => blockToMarkdown(c, indent + 2)).filter(Boolean).join("\n");
      return `${pad}<details>\n${pad}<summary>${summary}</summary>\n\n${inner}\n\n${pad}</details>`;
    }
    case "child_page": {
      const title = obj.title || "Untitled";
      // No stable URL without another API call; include ID for reference.
      return `${pad}- ${title} (child page: ${block.id})`;
    }
    default: {
      // Best-effort: if it has rich text, emit it; else emit a stub.
      if (txt) return (pad + txt + renderChildren(indent)).trimEnd();
      return `${pad}<!-- Unsupported block type: ${type} (${block.id}) -->`;
    }
  }
}

function blocksToMarkdown(blocks) {
  const parts = blocks.map((b) => blockToMarkdown(b, 0)).filter(Boolean);
  // Avoid excessive blank lines.
  return parts.join("\n\n").replace(/\n{3,}/g, "\n\n").trim() + "\n";
}

/** ---------- Markdown import ---------- **/

function parseMarkdownToBlocks(md) {
  const lines = String(md ?? "").replace(/\r\n/g, "\n").split("\n");

  const blocks = [];
  let i = 0;

  const pushParagraph = (buf) => {
    const text = buf.join("\n").trimEnd();
    if (!text) return;
    blocks.push({
      object: "block",
      type: "paragraph",
      paragraph: { rich_text: textToRichText(text) },
    });
  };

  while (i < lines.length) {
    const line = lines[i];

    // Code fence
    const codeStart = line.match(/^```(\w+)?\s*$/);
    if (codeStart) {
      const lang = codeStart[1] || "plain text";
      i++;
      const codeLines = [];
      while (i < lines.length && !lines[i].startsWith("```")) {
        codeLines.push(lines[i]);
        i++;
      }
      // Skip closing ```
      if (i < lines.length && lines[i].startsWith("```")) i++;

      blocks.push({
        object: "block",
        type: "code",
        code: { language: lang, rich_text: textToRichText(codeLines.join("\n")) },
      });
      continue;
    }

    // Divider
    if (/^\s*---\s*$/.test(line)) {
      blocks.push({ object: "block", type: "divider", divider: {} });
      i++;
      continue;
    }

    // Headings
    const h = line.match(/^(#{1,3})\s+(.*)$/);
    if (h) {
      const level = h[1].length;
      const text = h[2] ?? "";
      const type = level === 1 ? "heading_1" : level === 2 ? "heading_2" : "heading_3";
      blocks.push({
        object: "block",
        type,
        [type]: { rich_text: textToRichText(text) },
      });
      i++;
      continue;
    }

    // Todo
    const todo = line.match(/^\s*-\s+\[( |x|X)\]\s+(.*)$/);
    if (todo) {
      blocks.push({
        object: "block",
        type: "to_do",
        to_do: { checked: todo[1].toLowerCase() === "x", rich_text: textToRichText(todo[2] ?? "") },
      });
      i++;
      continue;
    }

    // Bulleted list
    const bullet = line.match(/^\s*[-*]\s+(.*)$/);
    if (bullet) {
      blocks.push({
        object: "block",
        type: "bulleted_list_item",
        bulleted_list_item: { rich_text: textToRichText(bullet[1] ?? "") },
      });
      i++;
      continue;
    }

    // Numbered list
    const num = line.match(/^\s*\d+\.\s+(.*)$/);
    if (num) {
      blocks.push({
        object: "block",
        type: "numbered_list_item",
        numbered_list_item: { rich_text: textToRichText(num[1] ?? "") },
      });
      i++;
      continue;
    }

    // Quote
    const quote = line.match(/^\s*>\s?(.*)$/);
    if (quote) {
      blocks.push({
        object: "block",
        type: "quote",
        quote: { rich_text: textToRichText(quote[1] ?? "") },
      });
      i++;
      continue;
    }

    // Blank line: paragraph boundary
    if (/^\s*$/.test(line)) {
      i++;
      continue;
    }

    // Paragraph: accumulate until blank line or a block marker.
    const buf = [];
    while (i < lines.length) {
      const l = lines[i];
      if (/^\s*$/.test(l)) break;
      if (/^```/.test(l)) break;
      if (/^\s*---\s*$/.test(l)) break;
      if (/^(#{1,3})\s+/.test(l)) break;
      if (/^\s*-\s+\[( |x|X)\]\s+/.test(l)) break;
      if (/^\s*[-*]\s+/.test(l)) break;
      if (/^\s*\d+\.\s+/.test(l)) break;
      if (/^\s*>\s?/.test(l)) break;

      buf.push(l);
      i++;
    }
    pushParagraph(buf);
  }

  return blocks;
}

/** ---------- append children ---------- **/

function chunk(arr, size) {
  const out = [];
  for (let i = 0; i < arr.length; i += size) out.push(arr.slice(i, i + size));
  return out;
}

async function appendBlocks(blockId, blocks) {
  const id = normaliseId(blockId);
  const chunks = chunk(blocks, MAX_BLOCKS_PER_APPEND);
  const results = [];

  for (const c of chunks) {
    const res = await notionRequest({
      method: "PATCH",
      path: `/v1/blocks/${id}/children`,
      body: { children: c },
    });
    results.push(res);
  }
  return results;
}

/** ---------- data sources and properties ---------- **/

async function getDataSource(dataSourceId) {
  const id = normaliseId(dataSourceId);
  return notionRequest({ method: "GET", path: `/v1/data_sources/${id}` });
}

function findTitlePropertyNameFromSchema(properties) {
  if (!isObject(properties)) return null;
  for (const [name, prop] of Object.entries(properties)) {
    if (isObject(prop) && prop.type === "title") return name;
  }
  return null;
}

function parseSetArgs(setArgs) {
  if (!setArgs) return [];
  const items = Array.isArray(setArgs) ? setArgs : [setArgs];
  return items.map((s) => {
    const idx = String(s).indexOf("=");
    if (idx === -1) throw new Error(`Invalid --set "${s}" (expected Prop=Value)`);
    const key = String(s).slice(0, idx).trim();
    const value = String(s).slice(idx + 1).trim();
    if (!key) throw new Error(`Invalid --set "${s}" (missing property name)`);
    return { key, value };
  });
}

function splitCommaList(s) {
  return String(s)
    .split(",")
    .map((x) => x.trim())
    .filter(Boolean);
}

function coercePrimitive(s) {
  const t = String(s).trim();
  if (t === "true") return true;
  if (t === "false") return false;
  if (t === "null") return null;
  if (/^-?\d+(\.\d+)?$/.test(t)) return Number(t);
  // Try JSON for objects/arrays only.
  if ((t.startsWith("{") && t.endsWith("}")) || (t.startsWith("[") && t.endsWith("]"))) {
    try { return JSON.parse(t); } catch { /* fallthrough */ }
  }
  return t;
}

function buildPropertyValue({ schemaProp, rawValue }) {
  const type = schemaProp.type;
  const v = coercePrimitive(rawValue);

  // Some property types are not settable via API in various endpoints.
  // Keep this conservative. If a user wants exotic types, they can extend this.
  if (["rollup", "formula", "created_by", "created_time", "last_edited_by", "last_edited_time"].includes(type)) {
    throw new Error(`Property type "${type}" cannot be set via the API`);
  }

  switch (type) {
    case "title":
      return { title: textToRichText(String(v ?? "")) };
    case "rich_text":
      return { rich_text: textToRichText(String(v ?? "")) };
    case "select":
      return { select: v ? { name: String(v) } : null };
    case "multi_select": {
      const names = Array.isArray(v) ? v.map(String) : splitCommaList(String(v ?? ""));
      return { multi_select: names.map((name) => ({ name })) };
    }
    case "status":
      return { status: v ? { name: String(v) } : null };
    case "date": {
      if (isObject(v)) return { date: v };
      return { date: v ? { start: String(v) } : null };
    }
    case "checkbox":
      return { checkbox: Boolean(v) };
    case "number":
      return { number: v === null ? null : Number(v) };
    case "url":
      return { url: v ? String(v) : null };
    case "email":
      return { email: v ? String(v) : null };
    case "phone_number":
      return { phone_number: v ? String(v) : null };
    case "people": {
      const ids = Array.isArray(v) ? v.map(String) : splitCommaList(String(v ?? ""));
      return { people: ids.map((id) => ({ id: normaliseId(id) })) };
    }
    case "relation": {
      const ids = Array.isArray(v) ? v.map(String) : splitCommaList(String(v ?? ""));
      return { relation: ids.map((id) => ({ id: normaliseId(id) })) };
    }
    default:
      throw new Error(`Unsupported property type "${type}" for --set`);
  }
}

function buildPropertiesFromSetArgs({ schema, setPairs }) {
  if (!schema || !isObject(schema.properties)) return {};
  const propsSchema = schema.properties;

  const out = {};
  for (const { key, value } of setPairs) {
    const schemaProp = propsSchema[key];
    if (!schemaProp) throw new Error(`Unknown property "${key}" on data source`);
    out[key] = buildPropertyValue({ schemaProp, rawValue: value });
  }
  return out;
}

/** ---------- commands ---------- **/

async function cmdWhoami({ compact }) {
  const me = await notionRequest({ method: "GET", path: "/v1/users/me" });
  print({ ok: true, user: me }, { compact });
}

async function cmdSearch({ compact, query, type = "all", limit = 20 }) {
  const q = String(query ?? "").trim();
  if (!q) throw new Error("search requires --query");

  const body = {
    query: q,
    sort: { direction: "descending", timestamp: "last_edited_time" },
    page_size: DEFAULT_PAGE_SIZE,
  };

  if (type !== "all") {
    if (type !== "page" && type !== "data_source") {
      throw new Error(`Invalid --type "${type}". Use page|data_source|all`);
    }
    body.filter = { property: "object", value: type };
  }

  const results = [];
  let cursor = undefined;

  while (results.length < limit) {
    if (cursor) body.start_cursor = cursor;

    const res = await notionRequest({ method: "POST", path: "/v1/search", body });
    if (res && Array.isArray(res.results)) results.push(...res.results);
    if (!res || !res.has_more) break;
    cursor = res.next_cursor;
    if (!cursor) break;
  }

  const trimmed = results.slice(0, limit).map((r) => {
    const base = {
      object: r.object,
      id: r.id,
      url: r.url,
      last_edited_time: r.last_edited_time,
    };
    if (r.object === "page") base.title = getPageTitle(r);
    if (r.object === "data_source") {
      // data sources have a title array similar to databases
      const title = Array.isArray(r.title) ? r.title.map((t) => t.plain_text ?? "").join("") : null;
      base.title = title;
    }
    base.parent = r.parent;
    return base;
  });

  print({ ok: true, query: q, type, results: trimmed }, { compact });
}

async function cmdGetPage({ compact, page }) {
  const id = normaliseId(page);
  const p = await notionRequest({ method: "GET", path: `/v1/pages/${id}` });
  print({ ok: true, id, title: getPageTitle(p), page: p }, { compact });
}

async function cmdExportMd({ compact, page, stdoutMd = false }) {
  const id = normaliseId(page);
  const pageObj = await notionRequest({ method: "GET", path: `/v1/pages/${id}` });
  const title = getPageTitle(pageObj);
  const blocks = await getBlockTree(id);
  const markdown = blocksToMarkdown(blocks);

  if (stdoutMd) {
    process.stdout.write(markdown);
    return;
  }

  print({ ok: true, id, title, markdown }, { compact });
}

async function readMarkdownInput({ md, mdFile, mdStdin }) {
  if (md !== undefined) return String(md);
  if (mdFile !== undefined) {
    const p = resolveHome(String(mdFile));
    return fs.readFileSync(p, "utf8");
  }
  if (mdStdin) {
    const chunks = [];
    for await (const chunk of process.stdin) chunks.push(chunk);
    return Buffer.concat(chunks).toString("utf8");
  }
  throw new Error("Provide markdown via --md, --md-file, or --md-stdin");
}

function buildParent({ parentPage, parentDataSource }) {
  if (parentPage) {
    const page_id = normaliseId(parentPage);
    return { type: "page_id", page_id };
  }
  if (parentDataSource) {
    const data_source_id = normaliseId(parentDataSource);
    return { type: "data_source_id", data_source_id };
  }
  throw new Error("Provide either --parent-page or --parent-data-source");
}

function buildPosition({ position, afterBlock }) {
  if (afterBlock) return { type: "after_block", after_block: { id: normaliseId(afterBlock) } };
  if (!position) return null;
  if (position !== "page_start" && position !== "page_end") {
    throw new Error('Invalid --position. Use "page_start" or "page_end"');
  }
  return { type: position };
}

function buildTemplate({ template, templateId }) {
  if (templateId) return { type: "template_id", template_id: normaliseId(templateId) };
  if (!template) return null;
  if (!["none", "default"].includes(template)) {
    throw new Error('Invalid --template. Use "none" or "default" (or use --template-id)');
  }
  return { type: template };
}

async function cmdCreateMd({ compact, title, parentPage, parentDataSource, md, mdFile, mdStdin, set, template, templateId, position, afterBlock }) {
  const t = String(title ?? "").trim();
  if (!t) throw new Error("create-md requires --title");

  const parent = buildParent({ parentPage, parentDataSource });
  const templateObj = buildTemplate({ template, templateId });
  const positionObj = buildPosition({ position, afterBlock });

  // Properties differ based on parent type.
  let properties = {};

  if (parent.type === "page_id") {
    properties = { title: { title: textToRichText(t) } };
  } else {
    const schema = await getDataSource(parent.data_source_id);
    const titlePropName = findTitlePropertyNameFromSchema(schema.properties) ?? "Name";
    properties[titlePropName] = { title: textToRichText(t) };

    const setPairs = parseSetArgs(set);
    const extra = buildPropertiesFromSetArgs({ schema, setPairs });
    properties = { ...properties, ...extra };
  }

  const body = {
    parent,
    properties,
  };

  if (parent.type === "page_id" && positionObj) body.position = positionObj;

  // Templates: children not allowed if using template (template is applied async).
  if (templateObj) {
    body.template = templateObj;
  } else {
    const markdown = await readMarkdownInput({ md, mdFile, mdStdin });
    const blocks = parseMarkdownToBlocks(markdown);
    if (blocks.length) body.children = blocks;
  }

  const res = await notionRequest({ method: "POST", path: "/v1/pages", body });
  print({ ok: true, created: res }, { compact });
}

async function cmdAppendMd({ compact, page, md, mdFile, mdStdin }) {
  const id = normaliseId(page);
  const markdown = await readMarkdownInput({ md, mdFile, mdStdin });
  const blocks = parseMarkdownToBlocks(markdown);

  const responses = await appendBlocks(id, blocks);
  print({ ok: true, page_id: id, appended_blocks: blocks.length, responses }, { compact });
}

async function cmdMove({ compact, page, toPage, toDataSource }) {
  const page_id = normaliseId(page);

  let parent = null;
  if (toPage) parent = { type: "page_id", page_id: normaliseId(toPage) };
  if (toDataSource) parent = { type: "data_source_id", data_source_id: normaliseId(toDataSource) };

  if (!parent) throw new Error("move requires --to-page or --to-data-source");

  const res = await notionRequest({
    method: "POST",
    path: `/v1/pages/${page_id}/move`,
    body: { parent },
  });

  print({ ok: true, moved_page_id: page_id, new_parent: parent, result: res }, { compact });
}

async function cmdListChildPages({ compact, page }) {
  const id = normaliseId(page);
  const blocks = await listBlockChildren(id);
  const childPages = blocks
    .filter((b) => b && b.type === "child_page")
    .map((b) => ({ id: b.id, title: b.child_page?.title ?? "Untitled" }));

  print({ ok: true, page_id: id, child_pages: childPages }, { compact });
}

function loadJsonFile(p) {
  const abs = resolveHome(String(p));
  const txt = fs.readFileSync(abs, "utf8");
  return JSON.parse(txt);
}

function ruleMatchesTitle(rule, title) {
  const m = rule.match;
  if (!m || !title) return false;
  if (m.title_regex) {
    const re = new RegExp(m.title_regex);
    return re.test(title);
  }
  if (m.contains) return title.toLowerCase().includes(String(m.contains).toLowerCase());
  return false;
}

async function cmdTriage({ compact, inboxPage, rules, limit = 50, apply = false }) {
  const inboxId = normaliseId(inboxPage);
  const rulesObj = loadJsonFile(rules);
  if (!Array.isArray(rulesObj)) throw new Error("rules JSON must be an array");

  const blocks = await listBlockChildren(inboxId);
  const childPages = blocks
    .filter((b) => b && b.type === "child_page")
    .map((b) => ({ id: b.id, title: b.child_page?.title ?? "Untitled" }))
    .slice(0, Number(limit));

  const plan = [];
  for (const p of childPages) {
    const rule = rulesObj.find((r) => ruleMatchesTitle(r, p.title));
    if (!rule) continue;

    const moveTo = rule.move_to;
    if (!moveTo || !moveTo.type || !moveTo.id) continue;

    plan.push({
      page_id: p.id,
      title: p.title,
      rule: rule.name ?? null,
      move_to: moveTo,
    });
  }

  if (!apply) {
    print({ ok: true, inbox_page: inboxId, apply: false, planned: plan }, { compact });
    return;
  }

  const results = [];
  for (const item of plan) {
    const parent =
      item.move_to.type === "page_id"
        ? { type: "page_id", page_id: normaliseId(item.move_to.id) }
        : item.move_to.type === "data_source_id"
          ? { type: "data_source_id", data_source_id: normaliseId(item.move_to.id) }
          : null;

    if (!parent) {
      results.push({ page_id: item.page_id, ok: false, error: `Unknown move_to.type: ${item.move_to.type}` });
      continue;
    }

    try {
      const res = await notionRequest({
        method: "POST",
        path: `/v1/pages/${normaliseId(item.page_id)}/move`,
        body: { parent },
      });
      results.push({ page_id: item.page_id, ok: true, moved_to: parent, result: res });
    } catch (e) {
      results.push({ page_id: item.page_id, ok: false, error: String(e) });
    }
  }

  print({ ok: true, inbox_page: inboxId, apply: true, moved: results }, { compact });
}

/** ---------- main ---------- **/

async function main() {
  const argv = process.argv.slice(2);
  if (argv.length === 0 || argv.includes("--help")) {
    usage();
    return;
  }

  const cmd = argv[0];
  const args = parseArgs(argv.slice(1));
  const compact = !!args.compact;

  try {
    switch (cmd) {
      case "whoami":
        await cmdWhoami({ compact });
        return;

      case "search":
        await cmdSearch({ compact, query: args.query, type: args.type ?? "all", limit: Number(args.limit ?? 20) });
        return;

      case "get-page":
        await cmdGetPage({ compact, page: args.page });
        return;

      case "export-md":
        await cmdExportMd({ compact, page: args.page, stdoutMd: !!args["stdout-md"] });
        return;

      case "create-md":
        await cmdCreateMd({
          compact,
          title: args.title,
          parentPage: args["parent-page"],
          parentDataSource: args["parent-data-source"],
          md: args.md,
          mdFile: args["md-file"],
          mdStdin: !!args["md-stdin"],
          set: args.set,
          template: args.template,
          templateId: args["template-id"],
          position: args.position,
          afterBlock: args["after-block"],
        });
        return;

      case "append-md":
        await cmdAppendMd({
          compact,
          page: args.page,
          md: args.md,
          mdFile: args["md-file"],
          mdStdin: !!args["md-stdin"],
        });
        return;

      case "move":
        await cmdMove({
          compact,
          page: args.page,
          toPage: args["to-page"],
          toDataSource: args["to-data-source"],
        });
        return;

      case "list-child-pages":
        await cmdListChildPages({ compact, page: args.page });
        return;

      case "triage":
        await cmdTriage({
          compact,
          inboxPage: args["inbox-page"],
          rules: args.rules,
          limit: Number(args.limit ?? 50),
          apply: !!args.apply,
        });
        return;

      default:
        throw new Error(`Unknown command: ${cmd}`);
    }
  } catch (e) {
    // If the error is JSON stringified by notionRequest, keep it as details.
    const s = String(e);
    let details = undefined;
    try {
      details = JSON.parse(s);
    } catch {
      details = s;
    }
    fail("Command failed", { details });
  }
}

await main();
