#!/usr/bin/env python3
"""nmap-pentest-scans script - AUTHORIZED SECURITY TESTING ONLY."""
from __future__ import annotations

import argparse
import json
import re
import sys
from datetime import datetime, timezone
from pathlib import Path


SHARED_DIR = Path(__file__).resolve().parents[2] / "autonomous-pentester" / "shared"
if str(SHARED_DIR) not in sys.path:
    sys.path.insert(0, str(SHARED_DIR))

from pentest_common import (  # noqa: E402
    load_payload,
    render_result,
    resolve_artifact_path,
    resolve_output_file,
    validate_scope,
    write_placeholder_artifact,
)


SKILL_NAME = "nmap-pentest-scans"
REPORT_STEM = "nmap-pentest-scans-report"
GENERATED_OUTPUTS = [
    "scan-plan.json",
    "scan-plan.md",
    "recommended-commands.txt",
    "findings/nmap-pentest-findings.json",
]


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description=(
            "Build scope-validated Nmap command plans for host discovery, service "
            "enumeration, and targeted script checks."
        )
    )
    parser.add_argument("--scope", default="scope.json")
    parser.add_argument("--target", required=True)
    parser.add_argument("--input", default=".")
    parser.add_argument("--output", default=".")
    parser.add_argument("--format", choices=["json", "md", "csv"], default="json")
    parser.add_argument("--dry-run", action="store_true")
    parser.add_argument("--i-have-authorization", action="store_true")
    parser.add_argument(
        "--profile",
        choices=["stealth", "balanced", "fast"],
        default="balanced",
        help="Scan profile for command tuning.",
    )
    parser.add_argument(
        "--include-udp",
        action="store_true",
        help="Include recommended UDP command in generated plan.",
    )
    return parser.parse_args()


def normalize_target_basename(target: str) -> str:
    """Normalize target values for safe filesystem basenames."""
    normalized = re.sub(r"[^A-Za-z0-9._-]+", "-", target.strip())
    normalized = normalized.strip("-._")
    return normalized or "target"


def build_command_plan(target: str, profile: str, include_udp: bool) -> list[dict[str, str]]:
    target_basename = normalize_target_basename(target)

    if profile == "stealth":
        timing = "-T2"
        rate = "--max-retries 2 --scan-delay 5ms"
    elif profile == "fast":
        timing = "-T4"
        rate = "--min-rate 500 --max-retries 1 --host-timeout 10m"
    else:
        timing = "-T3"
        rate = "--min-rate 100 --max-retries 2"

    plan: list[dict[str, str]] = [
        {
            "stage": "host-discovery",
            "command": (
                f"nmap -sn {target} -n --reason "
                f"-oA results/{target_basename}-01-discovery"
            ),
            "purpose": "Detect responsive hosts and collect probe reasons.",
        },
        {
            "stage": "tcp-triage",
            "command": (
                f"nmap -sS --top-ports 1000 {target} -n {timing} {rate} --reason "
                f"-oA results/{target_basename}-02-tcp-top1000"
            ),
            "purpose": "Find common exposed TCP services quickly.",
        },
        {
            "stage": "service-enum",
            "command": (
                f"nmap -sV -sC -p <OPEN_TCP_PORTS> {target} -n {timing} --reason "
                f"-oA results/{target_basename}-03-svc-default"
            ),
            "purpose": "Fingerprint service versions and run default safe scripts.",
        },
        {
            "stage": "web-tls",
            "command": (
                f"nmap -p 80,443 --script http-title,http-headers,http-methods,ssl-cert "
                f"{target} -n --script-timeout 30s "
                f"-oA results/{target_basename}-04-web-tls"
            ),
            "purpose": "Capture web fingerprinting and TLS certificate posture.",
        },
    ]

    if include_udp:
        plan.append(
            {
                "stage": "udp-followup",
                "command": (
                    f"nmap -sU --top-ports 200 {target} -n -T2 --reason "
                    f"-oA results/{target_basename}-05-udp-top200"
                ),
                "purpose": "Enumerate common UDP services after TCP baseline.",
            }
        )
    return plan


def build_finding(target: str) -> dict:
    now = datetime.now(timezone.utc).isoformat()
    return {
        "finding_id": f"{SKILL_NAME.replace('-', '_')}-001",
        "skill": SKILL_NAME,
        "timestamp": now,
        "target": target,
        "title": "Nmap reconnaissance identified externally reachable services",
        "cve": "N/A",
        "cwe": "CWE-200",
        "cvss_score": 6.5,
        "cvss_vector": "CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N",
        "owasp_category": "A05:2021 - Security Misconfiguration",
        "mitre_attack": "TA0007 - Discovery",
        "severity": "Medium",
        "description": "Service exposure was identified and should be reviewed against hardened baseline policy.",
        "proof_of_concept": "nmap -sS --top-ports 1000 <target> -n --reason",
        "screenshot": "assets/findings/placeholder.png",
        "remediation": "Restrict unnecessary network exposure and enforce service-level hardening controls.",
        "references": ["https://nmap.org/docs.html", "https://nmap.org/book/nse.html"],
        "status": "open",
    }


def write_plan_markdown(path: Path, target: str, profile: str, plan: list[dict[str, str]]) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    lines = [
        f"# Nmap Plan for {target}",
        "",
        f"- profile: {profile}",
        f"- generated_at: {datetime.now(timezone.utc).isoformat()}",
        "",
        "## Command Sequence",
    ]
    for item in plan:
        lines.append(f"- [{item['stage']}] `{item['command']}`")
        lines.append(f"  purpose: {item['purpose']}")
    path.write_text("\n".join(lines) + "\n", encoding="utf-8")


def write_command_list(path: Path, plan: list[dict[str, str]]) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    payload = "\n".join(item["command"] for item in plan) + "\n"
    path.write_text(payload, encoding="utf-8")


def main() -> int:
    args = parse_args()
    scope_ok, scope_meta = validate_scope(args.target, args.scope)
    report_path = resolve_output_file(args.output, args.format, REPORT_STEM)

    if not scope_ok:
        result = {
            "status": "error",
            "summary": "TARGET NOT IN AUTHORIZED SCOPE - ABORTING",
            "artifacts": [str(report_path)],
            "details": {
                "skill": SKILL_NAME,
                "target": args.target,
                "scope": scope_meta,
                "dry_run": args.dry_run,
            },
        }
        render_result(result, report_path, args.format)
        print(json.dumps(result, indent=2))
        return 1

    if not args.i_have_authorization and not args.dry_run:
        result = {
            "status": "error",
            "summary": "You must pass --i-have-authorization to confirm written authorization.",
            "artifacts": [str(report_path)],
            "details": {
                "skill": SKILL_NAME,
                "target": args.target,
                "scope": scope_meta,
                "dry_run": args.dry_run,
            },
        }
        render_result(result, report_path, args.format)
        print(json.dumps(result, indent=2))
        return 1

    payload = load_payload(args.input)
    plan = build_command_plan(args.target, args.profile, args.include_udp)
    finding = build_finding(args.target)
    artifacts: list[str] = []

    if not args.dry_run:
        plan_json = resolve_artifact_path(report_path.parent, "scan-plan.json")
        write_placeholder_artifact(
            plan_json,
            {
                "skill": SKILL_NAME,
                "target": args.target,
                "profile": args.profile,
                "generated_at": datetime.now(timezone.utc).isoformat(),
                "plan": plan,
                "input_payload": payload,
            },
        )
        artifacts.append(str(plan_json))

        plan_md = resolve_artifact_path(report_path.parent, "scan-plan.md")
        write_plan_markdown(plan_md, args.target, args.profile, plan)
        artifacts.append(str(plan_md))

        cmd_txt = resolve_artifact_path(report_path.parent, "recommended-commands.txt")
        write_command_list(cmd_txt, plan)
        artifacts.append(str(cmd_txt))

        findings_json = resolve_artifact_path(report_path.parent, "findings/nmap-pentest-findings.json")
        write_placeholder_artifact(
            findings_json,
            {
                "skill": SKILL_NAME,
                "target": args.target,
                "generated_at": datetime.now(timezone.utc).isoformat(),
                "findings": [finding],
                "plan_summary": [item["stage"] for item in plan],
            },
        )
        artifacts.append(str(findings_json))

    result = {
        "status": "ok",
        "summary": "Dry run completed" if args.dry_run else "Skill executed",
        "artifacts": artifacts + [str(report_path)],
        "details": {
            "skill": SKILL_NAME,
            "target": args.target,
            "scope": scope_meta,
            "profile": args.profile,
            "include_udp": args.include_udp,
            "plan": plan,
            "findings": [finding],
            "expected_outputs": GENERATED_OUTPUTS,
            "dry_run": args.dry_run,
        },
    }
    render_result(result, report_path, args.format)
    print(json.dumps(result, indent=2))
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
