# Nmap Scan Profiles

Use this reference to pick command templates quickly during authorized penetration tests.

## Table of Contents

1. Variables and Conventions
2. Baseline Setup
3. Host Discovery Profiles
4. Port Scan Profiles
5. Service and OS Fingerprinting
6. NSE Script Profiles
7. Firewall and Evasion Options
8. Performance and Reliability Tuning
9. Professional Runbooks
10. Output and Reporting
11. Common Pitfalls

## 1) Variables and Conventions

- Replace `<TARGET>` with host, CIDR, or range.
- Replace `<TARGETS_FILE>` with a newline-separated target list.
- Replace `<OUT>` with output prefix, for example `results/external-week1`.
- Run privileged scans as Administrator/root when required (`-sS`, raw packet features).
- Save outputs with `-oA <OUT>` whenever possible.

## 2) Baseline Setup

```bash
# Version check
nmap --version

# Basic syntax check on a target list
nmap -iL <TARGETS_FILE> -sn -n --reason -oA <OUT>-discovery
```

Use `-n` to skip DNS when speed and signal fidelity matter.

## 3) Host Discovery Profiles

```bash
# Standard ICMP + common probes
nmap -sn <TARGET> -n --reason -oA <OUT>-host-discovery

# ARP discovery on local network segment
nmap -sn -PR <TARGET> -n --reason -oA <OUT>-arp-discovery

# Discovery with TCP SYN ping to common service ports
nmap -sn -PS22,80,443,3389,445 <TARGET> -n --reason -oA <OUT>-syn-ping

# Discovery with TCP ACK ping
nmap -sn -PA80,443,3389 <TARGET> -n --reason -oA <OUT>-ack-ping

# Discovery with UDP ping
nmap -sn -PU53,67,68,123,161 <TARGET> -n --reason -oA <OUT>-udp-ping

# Treat all hosts as up (when ping is blocked)
nmap -Pn -sn <TARGET> -n --reason -oA <OUT>-no-ping
```

Use `-Pn` carefully because it increases scan time/noise on dead hosts.

## 4) Port Scan Profiles

### Fast Triage

```bash
# Top TCP ports, fast initial map
nmap -sS --top-ports 1000 <TARGET> -n -T3 --reason -oA <OUT>-tcp-top1000

# Fast scan preset
nmap -sS -F <TARGET> -n -T3 --reason -oA <OUT>-tcp-fast
```

### Full Coverage

```bash
# Full TCP port sweep
nmap -sS -p- <TARGET> -n -T3 --reason -oA <OUT>-tcp-all

# UDP common ports
nmap -sU --top-ports 200 <TARGET> -n -T2 --reason -oA <OUT>-udp-top200

# Combined TCP + UDP (costly)
nmap -sS -sU -p T:1-1000,U:53,67,68,69,123,137,138,139,161,162 <TARGET> -n -T2 --reason -oA <OUT>-tcp-udp-mixed
```

### Firewall State Mapping

```bash
# ACK scan to infer filtering
nmap -sA -p 22,80,443,3389 <TARGET> -n --reason -oA <OUT>-ack-map

# Window scan (stack-dependent)
nmap -sW -p 22,80,443 <TARGET> -n --reason -oA <OUT>-window-scan

# Maimon scan (niche behavior)
nmap -sM -p 80,443 <TARGET> -n --reason -oA <OUT>-maimon
```

### Non-SYN Alternatives

```bash
# TCP connect scan (no raw sockets needed)
nmap -sT -p- <TARGET> -n -T3 --reason -oA <OUT>-connect-all

# NULL / FIN / Xmas (mainly for stateless filtering checks)
nmap -sN -p 1-1024 <TARGET> -n --reason -oA <OUT>-null
nmap -sF -p 1-1024 <TARGET> -n --reason -oA <OUT>-fin
nmap -sX -p 1-1024 <TARGET> -n --reason -oA <OUT>-xmas
```

## 5) Service and OS Fingerprinting

```bash
# Service/version on known open ports
nmap -sV -p 22,80,443,445,3389 <TARGET> -n --version-intensity 7 --reason -oA <OUT>-svc

# Aggressive service probing
nmap -sV --version-all <TARGET> -n --reason -oA <OUT>-svc-all

# OS detection + traceroute
nmap -O --osscan-guess --traceroute <TARGET> -n --reason -oA <OUT>-os-trace

# Aggressive bundle (OS, version, scripts, traceroute)
nmap -A <TARGET> -n -T3 --reason -oA <OUT>-aggressive
```

Use `-A` on limited targets first because it can be noisy.

## 6) NSE Script Profiles

Prefer service-targeted scripts over broad categories when possible.

### Safe Baseline

```bash
# Default scripts on discovered services
nmap -sC -sV <TARGET> -n -T3 --reason -oA <OUT>-default-scripts

# Safe script category only
nmap --script safe -sV <TARGET> -n --reason -oA <OUT>-safe-category
```

### Web

```bash
nmap -p80,443 --script http-title,http-headers,http-methods,http-enum,http-server-header <TARGET> -n -oA <OUT>-http-enum
nmap -p443 --script ssl-cert,ssl-enum-ciphers,ssl-dh-params <TARGET> -n -oA <OUT>-tls-audit
```

### SMB / Windows

```bash
nmap -p139,445 --script smb-os-discovery,smb-protocols,smb-security-mode,smb2-security-mode <TARGET> -n -oA <OUT>-smb-posture
nmap -p445 --script smb-enum-shares,smb-enum-users,smb-enum-domains <TARGET> -n -oA <OUT>-smb-enum
```

### DNS, SNMP, LDAP, RDP, NTP

```bash
nmap -p53 --script dns-recursion,dns-nsid,dns-service-discovery <TARGET> -n -oA <OUT>-dns
nmap -p161 --script snmp-info,snmp-interfaces,snmp-processes,snmp-sysdescr <TARGET> -n -oA <OUT>-snmp
nmap -p389,636 --script ldap-rootdse,ldap-search <TARGET> -n -oA <OUT>-ldap
nmap -p3389 --script rdp-enum-encryption,rdp-ntlm-info <TARGET> -n -oA <OUT>-rdp
nmap -p123 --script ntp-info,ntp-monlist <TARGET> -n -oA <OUT>-ntp
```

### Auth and Brute Categories (High Risk, Explicit Approval)

```bash
# Run only when scope explicitly allows authentication testing
nmap --script auth -p 21,22,80,443,445 <TARGET> -n -oA <OUT>-auth-category
nmap --script brute -p 21,22,80,443,445 <TARGET> -n --script-args brute.firstonly=true -oA <OUT>-brute-category
```

### Vuln Category (High Risk, Explicit Approval)

```bash
# Scope this to specific hosts/services after baseline confirmation
nmap --script vuln -sV <TARGET> -n -T2 -oA <OUT>-vuln-category
```

Use `--script-args` for credentialed or tuned checks when needed.

## 7) Firewall and Evasion Options

Use only with written authorization and defined rules of engagement.

```bash
# Fragment packets
nmap -sS -f <TARGET> -n -oA <OUT>-frag

# Custom MTU (must be multiple of 8)
nmap -sS --mtu 24 <TARGET> -n -oA <OUT>-mtu

# Decoy scan
nmap -sS -D RND:5 <TARGET> -n -oA <OUT>-decoy

# Source port manipulation
nmap -sS --source-port 53 <TARGET> -n -oA <OUT>-srcport53

# Spoof MAC (local segment only)
nmap -sS --spoof-mac 0 <TARGET> -n -oA <OUT>-spoof-mac

# Append random payload bytes
nmap -sS --data-length 32 <TARGET> -n -oA <OUT>-data-length
```

If evasion is authorized, document business justification and expected impact.

## 8) Performance and Reliability Tuning

```bash
# Conservative and stealthier
nmap -sS --top-ports 1000 <TARGET> -n -T2 --max-retries 2 --scan-delay 5ms -oA <OUT>-slow-steady

# Balanced enterprise scan
nmap -sS --top-ports 1000 <TARGET> -n -T3 --min-rate 100 --max-retries 2 -oA <OUT>-balanced

# Faster scan in resilient environments
nmap -sS --top-ports 1000 <TARGET> -n -T4 --min-rate 500 --max-retries 1 --host-timeout 10m -oA <OUT>-fast
```

Useful options:

- `--min-rate` and `--max-rate` to control packet throughput.
- `--min-hostgroup` and `--max-hostgroup` to tune parallel hosts.
- `--min-parallelism` and `--max-parallelism` for probe concurrency.
- `--max-rtt-timeout` and `--initial-rtt-timeout` for high latency paths.

## 9) Professional Runbooks

### External Perimeter (Breadth First)

```bash
nmap -sn <TARGET> -n --reason -oA <OUT>-01-discovery
nmap -sS --top-ports 1000 <TARGET> -n -T3 --reason -oA <OUT>-02-tcp-top
nmap -sV -p <OPEN_TCP_PORTS> <TARGET> -n --reason -oA <OUT>-03-version
nmap --script safe -p <OPEN_TCP_PORTS> <TARGET> -n -oA <OUT>-04-safe-scripts
```

### Internal Network (Depth on Live Hosts)

```bash
nmap -sn -PR <TARGET> -n --reason -oA <OUT>-01-arp-discovery
nmap -sS -p- <TARGET> -n -T3 --reason -oA <OUT>-02-tcp-all
nmap -sU --top-ports 200 <TARGET> -n -T2 --reason -oA <OUT>-03-udp-top
nmap -sV -O --traceroute <TARGET> -n -T3 --reason -oA <OUT>-04-fingerprint
```

### Active Directory Focus

```bash
nmap -p53,88,135,139,389,445,464,636,3268,3269,3389 -sS -sV <TARGET> -n -T3 -oA <OUT>-ad-core
nmap -p389,636 --script ldap-rootdse,ldap-search <TARGET> -n -oA <OUT>-ad-ldap
nmap -p445 --script smb-os-discovery,smb-security-mode,smb2-security-mode,smb-enum-shares <TARGET> -n -oA <OUT>-ad-smb
```

### Web and TLS Focus

```bash
nmap -p80,443,8080,8443 -sS -sV <TARGET> -n -T3 -oA <OUT>-web-ports
nmap -p80,443,8080,8443 --script http-title,http-enum,http-methods,http-security-headers <TARGET> -n -oA <OUT>-web-scripts
nmap -p443,8443 --script ssl-cert,ssl-enum-ciphers,ssl-dh-params <TARGET> -n -oA <OUT>-tls-scripts
```

## 10) Output and Reporting

```bash
# Save all major formats
nmap -sS -sV <TARGET> -n --reason -oA <OUT>

# Human-readable
nmap -sS -sV <TARGET> -n --reason -oN <OUT>.nmap

# XML for parsers/automation
nmap -sS -sV <TARGET> -n --reason -oX <OUT>.xml

# Grepable format for quick extraction
nmap -sS -sV <TARGET> -n --reason -oG <OUT>.gnmap
```

Quick extraction examples:

```bash
# Open ports from gnmap output
grep "/open/" <OUT>.gnmap

# Open TCP services summary
grep "/open/tcp/" <OUT>.gnmap | cut -d " " -f 2-
```

## 11) Common Pitfalls

- Scanning UDP too broadly without timing controls can stall assessments.
- Using `-Pn` on large ranges can create excessive noise and long runtimes.
- Running intrusive NSE categories before baseline reconnaissance can break change windows.
- Interpreting `open|filtered` as open without corroborating evidence leads to false positives.
- Forgetting `-oA` impairs reproducibility and auditability.
