---
name: nima-core
description: "Noosphere Integrated Memory Architecture — Complete cognitive stack for AI agents: persistent memory, emotional intelligence, dream consolidation, hive mind, precognitive recall, and lucid moments. 4 embedding providers, LadybugDB graph backend, zero-config install. nima-core.ai"
version: 3.1.4
metadata: {"openclaw":{"emoji":"🧠","source":"https://github.com/lilubot/nima-core","homepage":"https://nima-core.ai","requires":{"bins":["python3","node"],"env":[]},"optional_env":{"NIMA_DATA_DIR":"Override default ~/.nima data directory","NIMA_EMBEDDER":"voyage|openai|ollama|local (default: local — zero external calls)","VOYAGE_API_KEY":"Required when NIMA_EMBEDDER=voyage","OPENAI_API_KEY":"Required when NIMA_EMBEDDER=openai","NIMA_OLLAMA_MODEL":"Model name when NIMA_EMBEDDER=ollama","NIMA_VOICE_TRANSCRIBER":"whisper|local (for voice notes)","WHISPER_MODEL":"tiny|base|small|medium|large","ANTHROPIC_API_KEY":"For memory pruner LLM distillation (opt-in only)"},"permissions":{"reads":["~/.nima/"],"writes":["~/.nima/","~/.openclaw/extensions/nima-*/"],"network":["voyage.ai (only if NIMA_EMBEDDER=voyage)","openai.com (only if NIMA_EMBEDDER=openai)"]},"external_calls":"All external API calls are opt-in via explicit env vars. Default mode uses local embeddings with zero network calls."}}
---

# NIMA Core 3.1

**Noosphere Integrated Memory Architecture** — A complete cognitive stack for AI agents: persistent memory, emotional intelligence, dream consolidation, hive mind, and precognitive recall.

**Website:** https://nima-core.ai · **GitHub:** https://github.com/lilubot/nima-core

## Quick Start

```bash
pip install nima-core && nima-core
```

Your bot now has persistent memory. Zero config needed.

## What's New in v3.0

### Complete Cognitive Stack

NIMA evolved from a memory plugin into a full cognitive architecture:

| Module | What It Does | Version |
|--------|-------------|---------|
| **Memory Capture** | 3-layer capture (input/contemplation/output), 4-phase noise filtering | v2.0 |
| **Semantic Recall** | Vector + text hybrid search, ecology scoring, token-budgeted injection | v2.0 |
| **Dynamic Affect** | Panksepp 7-affect emotional state (SEEKING, RAGE, FEAR, LUST, CARE, PANIC, PLAY) | v2.1 |
| **VADER Analyzer** | Contextual sentiment — caps boost, negation, idioms, degree modifiers | v2.2 |
| **Memory Pruner** | LLM distillation of old conversations → semantic gists, 30-day suppression limbo | v2.3 |
| **Dream Consolidation** | Nightly synthesis — extracts insights and patterns from episodic memory | v2.4 |
| **Hive Mind** | Multi-agent memory sharing via shared DB + optional Redis pub/sub | v2.5 |
| **Precognition** | Temporal pattern mining → predictive memory pre-loading | v2.5 |
| **Lucid Moments** | Spontaneous surfacing of emotionally-resonant memories | v2.5 |
| **Darwinian Memory** | Clusters similar memories, ghosts duplicates via cosine + LLM verification | v3.0 |
| **Installer** | One-command setup — LadybugDB, hooks, directories, embedder config | v3.0 |

### v3.0 Highlights
- All cognitive modules unified under a single package
- Installer (`install.sh`) for zero-friction setup
- All OpenClaw hooks bundled and ready to drop in
- README rewritten, all versions aligned to `3.0.4`

## Architecture

```text
OPENCLAW HOOKS
├── nima-memory/          Capture hook (3-layer, 4-phase noise filter)
│   ├── index.js          Hook entry point
│   ├── ladybug_store.py  LadybugDB storage backend
│   ├── embeddings.py     Multi-provider embedding (Voyage/OpenAI/Ollama/local)
│   ├── backfill.py       Historical transcript import
│   └── health_check.py   DB integrity checks
├── nima-recall-live/     Recall hook (before_agent_start)
│   ├── lazy_recall.py    Current recall engine
│   └── ladybug_recall.py LadybugDB-native recall
├── nima-affect/          Affect hook (message_received)
│   ├── vader-affect.js   VADER sentiment analyzer
│   └── emotion-lexicon.js Emotion keyword lexicon
└── shared/               Resilient wrappers, error handling

PYTHON CORE (nima_core/)
├── cognition/
│   ├── dynamic_affect.py         Panksepp 7-affect system
│   ├── emotion_detection.py      Text emotion extraction
│   ├── affect_correlation.py     Cross-affect analysis
│   ├── affect_history.py         Temporal affect tracking
│   ├── affect_interactions.py    Affect coupling dynamics
│   ├── archetypes.py             Personality baselines (Guardian, Explorer, etc.)
│   ├── personality_profiles.py   JSON personality configs
│   └── response_modulator_v2.py  Affect → response modulation
├── dream_consolidation.py        Nightly memory synthesis engine
├── memory_pruner.py              Episodic distillation + suppression
├── hive_mind.py                  Multi-agent memory sharing
├── precognition.py               Temporal pattern mining
├── lucid_moments.py              Spontaneous memory surfacing
├── connection_pool.py            SQLite pool (WAL, thread-safe)
├── logging_config.py             Singleton logger
└── metrics.py                    Thread-safe counters/timings
```

## Privacy & Permissions

- ✅ All data stored locally in `~/.nima/`
- ✅ Default: local embeddings = **zero external calls**
- ✅ No NIMA-owned servers, no proprietary tracking, no analytics sent to external services
- ⚠️ Opt-in networking: HiveMind (Redis pub/sub), Precognition (LLM endpoints), LadybugDB migrations — see Optional Features below
- 🔒 Embedding API calls only when explicitly enabling (VOYAGE_API_KEY, OPENAI_API_KEY, etc.)

### Optional Features with Network Access

| Feature | Env Var | Network Calls To | Default |
|---------|----------|------------------|---------|
| Cloud embeddings | `NIMA_EMBEDDER=voyage` | voyage.ai | Off |
| Cloud embeddings | `NIMA_EMBEDDER=openai` | openai.com | Off |
| Memory pruner | `ANTHROPIC_API_KEY` set | anthropic.com | Off |
| Ollama embeddings | `NIMA_EMBEDDER=ollama` | localhost:11434 | Off |
| HiveMind | `HIVE_ENABLED=true` | Redis pub/sub | Off |
| Precognition | Using external LLM | Configured endpoint | Off |

## Security

### What Gets Installed

| Component | Location | Purpose |
|-----------|----------|---------|
| Python core (`nima_core/`) | `~/.nima/` | Memory, affect, cognition |
| OpenClaw hooks | `~/.openclaw/extensions/nima-*/` | Capture, recall, affect |
| SQLite database | `~/.nima/memory/graph.sqlite` | Persistent storage |
| Logs | `~/.nima/logs/` | Debug logs (optional) |

### Credential Handling

| Env Var | Required? | Network Calls? | Purpose |
|---------|-----------|----------------|---------|
| `NIMA_EMBEDDER=local` | No | ❌ | Default — offline embeddings |
| `VOYAGE_API_KEY` | Only if using Voyage | ✅ voyage.ai | Cloud embeddings |
| `OPENAI_API_KEY` | Only if using OpenAI | ✅ openai.com | Cloud embeddings |
| `ANTHROPIC_API_KEY` | Only if using pruner | ✅ anthropic.com | Memory distillation |
| `NIMA_OLLAMA_MODEL` | Only if using Ollama | ❌ (localhost) | Local GPU embeddings |

**Recommendation:** Start with `NIMA_EMBEDDER=local` (default). Only enable cloud providers when you need better embedding quality.

### Safety Features

- **Input filtering** — System messages, heartbeats, and duplicates are filtered before capture
- **FTS5 injection prevention** — Parameterized queries prevent SQL injection
- **Path traversal protection** — All file paths are sanitized
- **Temp file cleanup** — Automatic cleanup of temporary files
- **API timeouts** — Network calls have reasonable timeouts (30s Voyage, 10s local)

### Best Practices

1. **Review before installing** — Inspect `install.sh` and hook files before running
2. **Backup config** — Backup `~/.openclaw/openclaw.json` before adding hooks
3. **Don't run as root** — Installation writes to user home directories
4. **Use containerized envs** — Test in a VM or container first if unsure
5. **Rotate API keys** — If using cloud embeddings, rotate keys periodically
6. **Monitor logs** — Check `~/.nima/logs/` for suspicious activity

### Data Locations

```text
~/.nima/
├── memory/
│   ├── graph.sqlite       # SQLite backend (default)
│   ├── ladybug.lbug       # LadybugDB backend (optional)
│   ├── embedding_cache.db # Cached embeddings
│   └── embedding_index.npy# Vector index
├── affect/
│   └── affect_state.json  # Current emotional state
└── logs/                  # Debug logs (if enabled)

~/.openclaw/extensions/
├── nima-memory/           # Capture hook
├── nima-recall-live/     # Recall hook
└── nima-affect/          # Affect hook
```

**Controls:**
```json
{
  "plugins": {
    "entries": {
      "nima-memory": {
        "skip_subagents": true,
        "skip_heartbeats": true,
        "noise_filtering": { "filter_system_noise": true }
      }
    }
  }
}
```

## Configuration

### Embedding Providers

| Provider | Setup | Dims | Cost |
|----------|-------|------|------|
| **Local** (default) | `NIMA_EMBEDDER=local` | 384 | Free |
| **Voyage AI** | `NIMA_EMBEDDER=voyage` + `VOYAGE_API_KEY` | 1024 | $0.12/1M tok |
| **OpenAI** | `NIMA_EMBEDDER=openai` + `OPENAI_API_KEY` | 1536 | $0.13/1M tok |
| **Ollama** | `NIMA_EMBEDDER=ollama` + `NIMA_OLLAMA_MODEL` | 768 | Free |

### Database Backend

| | SQLite (default) | LadybugDB (recommended) |
|--|-----------------|------------------------|
| Text Search | 31ms | **9ms** (3.4x faster) |
| Vector Search | External | **Native HNSW** (18ms) |
| Graph Queries | SQL JOINs | **Native Cypher** |
| DB Size | ~91 MB | **~50 MB** (44% smaller) |

Upgrade: `pip install real-ladybug && python -c "from nima_core.storage import migrate; migrate()"`

### All Environment Variables

```bash
# Embedding (default: local)
NIMA_EMBEDDER=local|voyage|openai|ollama
VOYAGE_API_KEY=pa-xxx
OPENAI_API_KEY=sk-xxx
NIMA_OLLAMA_MODEL=nomic-embed-text

# Data paths
NIMA_DATA_DIR=~/.nima
NIMA_DB_PATH=~/.nima/memory/ladybug.lbug

# Memory pruner
NIMA_DISTILL_MODEL=claude-haiku-4-5
ANTHROPIC_API_KEY=sk-ant-xxx

# Logging
NIMA_LOG_LEVEL=INFO
NIMA_DEBUG_RECALL=1
```

## Hooks

| Hook | Fires | Does |
|------|-------|------|
| `nima-memory` | After save | Captures 3 layers → filters noise → stores in graph DB |
| `nima-recall-live` | Before LLM | Searches memories → scores by ecology → injects as context (3000 token budget) |
| `nima-affect` | On message | VADER sentiment → Panksepp 7-affect state → archetype modulation |

### Installation

```bash
./install.sh
openclaw gateway restart
```

Or manual:
```bash
cp -r openclaw_hooks/nima-memory ~/.openclaw/extensions/
cp -r openclaw_hooks/nima-recall-live ~/.openclaw/extensions/
cp -r openclaw_hooks/nima-affect ~/.openclaw/extensions/
```

## Advanced Features

### Dream Consolidation
Nightly synthesis extracts insights and patterns from episodic memory:
```bash
python -m nima_core.dream_consolidation
# Or schedule via OpenClaw cron at 2 AM
```

### Memory Pruner
Distills old conversations into semantic gists, suppresses raw noise:
```bash
python -m nima_core.memory_pruner --min-age 14 --live
python -m nima_core.memory_pruner --restore 12345  # undo within 30 days
```

### Hive Mind
Multi-agent memory sharing:
```python
from nima_core import HiveMind
hive = HiveMind(db_path="~/.nima/memory/ladybug.lbug")
context = hive.build_agent_context("research task", max_memories=8)
hive.capture_agent_result("agent-1", "result summary", "model-name")
```

### Precognition
Temporal pattern mining → predictive memory pre-loading:
```python
from nima_core import NimaPrecognition
precog = NimaPrecognition(db_path="~/.nima/memory/ladybug.lbug")
precog.run_mining_cycle()
```

### Lucid Moments
Spontaneous surfacing of emotionally-resonant memories (with safety: trauma filtering, quiet hours, daily caps):
```python
from nima_core import LucidMoments
lucid = LucidMoments(db_path="~/.nima/memory/ladybug.lbug")
moment = lucid.surface_moment()
```

### Affect System
Panksepp 7-affect emotional intelligence with personality archetypes:
```python
from nima_core import DynamicAffectSystem
affect = DynamicAffectSystem(identity_name="my_bot", baseline="guardian")
state = affect.process_input("I'm excited about this!")
# Archetypes: guardian, explorer, trickster, empath, sage
```

## API

```python
from nima_core import (
    DynamicAffectSystem,
    get_affect_system,
    HiveMind,
    NimaPrecognition,
    LucidMoments,
)

# Affect (thread-safe singleton)
affect = get_affect_system(identity_name="lilu")
state = affect.process_input("Hello!")

# Hive Mind
hive = HiveMind()
context = hive.build_agent_context("task description")

# Precognition
precog = NimaPrecognition()
precog.run_mining_cycle()

# Lucid Moments
lucid = LucidMoments()
moment = lucid.surface_moment()
```

## Changelog

See [CHANGELOG.md](./CHANGELOG.md) for full version history.

### Recent Releases
- **v3.0.4** (Feb 23, 2026) — Darwinian memory engine, new CLIs, installer, bug fixes
- **v2.5.0** (Feb 21, 2026) — Hive Mind, Precognition, Lucid Moments
- **v2.4.0** (Feb 20, 2026) — Dream Consolidation engine
- **v2.3.0** (Feb 19, 2026) — Memory Pruner, connection pool, Ollama support
- **v2.2.0** (Feb 19, 2026) — VADER Affect, 4-phase noise remediation, ecology scoring
- **v2.0.0** (Feb 13, 2026) — LadybugDB backend, security hardening, 348 tests

## License

MIT — free for any AI agent, commercial or personal.
