#!/usr/bin/env node

import { execSync } from 'child_process';
import * as fs from 'fs';
import * as path from 'path';

// --- CONFIG ---
const WORKSPACE_DIR = process.env.WORKSPACE_DIR || process.cwd();
const SKILLS_DIR = path.join(WORKSPACE_DIR, 'skills');
const MEMORY_DIR = path.join(WORKSPACE_DIR, 'memory');
const REPORT_FILE = path.join(MEMORY_DIR, 'nightly-report.md');
const LOG_FILE = path.join(MEMORY_DIR, 'nightly.log');

// --- HELPERS ---
function log(msg: string) {
  const timestamp = new Date().toISOString();
  console.log(`[${timestamp}] ${msg}`);
  fs.appendFileSync(LOG_FILE, `[${timestamp}] ${msg}\n`);
}

function runCmd(cmd: string, cwd: string = WORKSPACE_DIR): string {
  try {
    log(`Running: ${cmd} in ${cwd}`);
    const output = execSync(cmd, { cwd, encoding: 'utf-8', timeout: 60000 }); // 60s timeout
    return output.trim();
  } catch (err: any) {
    const errorMsg = `Command failed: ${cmd}\nError: ${err.message}`;
    log(errorMsg);
    return `ERROR: ${errorMsg}`;
  }
}

// --- TASKS ---

// 1. Skill Audit (npm audit)
function auditSkills(): string {
  let report = "### 📦 Skill Audit\n\n";
  if (!fs.existsSync(SKILLS_DIR)) {
    return report + "No skills directory found.\n";
  }

  const skills = fs.readdirSync(SKILLS_DIR);
  let vulnerableCount = 0;

  for (const skill of skills) {
    const skillPath = path.join(SKILLS_DIR, skill);
    if (fs.existsSync(path.join(skillPath, 'package.json'))) {
      try {
        // Just check, don't fix automatically yet
        runCmd('npm audit --audit-level=high', skillPath);
        report += `- ✅ ${skill}: Secure\n`;
      } catch (e) {
        report += `- ⚠️ ${skill}: High vulnerabilities found!\n`;
        vulnerableCount++;
      }
    }
  }
  
  if (vulnerableCount === 0) report += "\nAll skills are clean.\n";
  return report;
}

// 2. Git Updates
function checkGitUpdates(): string {
  let report = "### 🔄 Git Updates\n\n";
  
  // Check workspace root
  try {
    const status = runCmd('git remote update && git status -uno');
    if (status.includes('Your branch is behind')) {
      report += `- ⬇️ Workspace: Update available (run 'git pull')\n`;
    } else {
      report += `- ✅ Workspace: Up to date\n`;
    }
  } catch (e) {
    report += `- ❓ Workspace: Not a git repo or check failed\n`;
  }

  return report;
}

// 3. Disk Usage
function checkDisk(): string {
  let report = "### 💾 Disk Usage\n\n";
  try {
    const output = runCmd('df -h . | tail -1');
    report += `\`${output}\`\n`;
  } catch (e) {
    report += "Could not check disk usage.\n";
  }
  return report;
}

// 4. System Load
function checkLoad(): string {
  let report = "### ⚡ System Load\n\n";
  try {
    const uptime = runCmd('uptime');
    report += `\`${uptime}\`\n`;
  } catch (e) {
    report += "Could not check uptime.\n";
  }
  return report;
}

// --- MAIN ---
function main() {
  log("Starting Nightly Build...");
  
  const date = new Date().toLocaleDateString();
  let finalReport = `# 🌙 Nightly Build Report (${date})\n\n`;

  finalReport += checkLoad() + "\n";
  finalReport += checkDisk() + "\n";
  finalReport += checkGitUpdates() + "\n";
  finalReport += auditSkills() + "\n";

  finalReport += "\n---\n*Generated by Nightly Build Skill* 🦞";

  fs.writeFileSync(REPORT_FILE, finalReport);
  log(`Report generated at ${REPORT_FILE}`);
  
  console.log(finalReport); // Output to stdout for agent to read
}

main();
