#!/bin/bash
#!/bin/bash

set -e # Exit immediately if a command exits with a non-zero status.

# --- Default values ---
SERVICE_NAME=""
DOMAIN_NAME=""
PORT=""
CONFIG_PATH=""
CONTAINER_NAME=""

# --- Parse Command-Line Arguments ---
while [[ "$#" -gt 0 ]]; do
    case $1 in
        --service-name) SERVICE_NAME="$2"; shift ;;
        --domain) DOMAIN_NAME="$2"; shift ;;
        --port) PORT="$2"; shift ;;
        --config-path) CONFIG_PATH="$2"; shift ;;
        --container-name) CONTAINER_NAME="$2"; shift ;;
        *) echo "Unknown parameter passed: $1"; exit 1 ;;
    esac
    shift
done

# --- Validate Core Arguments ---
if [ -z "$SERVICE_NAME" ] || [ -z "$DOMAIN_NAME" ] || [ -z "$PORT" ]; then
    echo "Error: --service-name, --domain, and --port are required." >&2
    exit 1
fi

# --- Environment Awareness: Use environment variables as fallback ---
CONFIG_PATH=${CONFIG_PATH:-$NGINX_CONFIG_PATH}
CONTAINER_NAME=${CONTAINER_NAME:-$NGINX_CONTAINER_NAME}

if [ -z "$CONFIG_PATH" ]; then
    echo "Error: Nginx config path is not set. Use --config-path or set NGINX_CONFIG_PATH environment variable." >&2
    exit 1
fi

if [ -z "$CONTAINER_NAME" ]; then
    echo "Error: Nginx container name is not set. Use --container-name or set NGINX_CONTAINER_NAME environment variable." >&2
    exit 1
fi

# --- Main Logic ---
FULL_DOMAIN="${SERVICE_NAME}.${DOMAIN_NAME}"
CONF_FILE_PATH="${CONFIG_PATH}/${SERVICE_NAME}.conf"

echo "--- Operation Summary ---"
echo "Service Name: $SERVICE_NAME"
echo "Full Domain: $FULL_DOMAIN"
echo "Target Port: $PORT"
echo "Config File: $CONF_FILE_PATH"
echo "Nginx Container: $CONTAINER_NAME"
echo "-------------------------"

# Create Nginx config content
NGINX_CONF=$(cat <<EOF
server {
    listen 80;
    server_name ${FULL_DOMAIN};

    location / {
        proxy_pass http://127.0.0.1:${PORT};
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection "upgrade";
    }
}
EOF
)

echo "Writing config to ${CONF_FILE_PATH}..."
echo "$NGINX_CONF" > "$CONF_FILE_PATH"
echo "[OK] Config file written."

# --- Safety Check (熔断机制) ---
echo "Testing Nginx configuration..."
if docker exec "$CONTAINER_NAME" nginx -t &> /tmp/nginx_test_output.log; then
    echo "[OK] Configuration test successful."
    
    # --- Apply Configuration ---
    echo "Reloading Nginx service..."
    if docker exec "$CONTAINER_NAME" nginx -s reload; then
        echo "[SUCCESS] Nginx reloaded successfully. Service '${SERVICE_NAME}' is now live at http://${FULL_DOMAIN}"
    else
        echo "Error: Failed to reload Nginx. Rolling back..." >&2
        rm -f "$CONF_FILE_PATH"
        echo "[ROLLED BACK] Deleted ${CONF_FILE_PATH}" >&2
        exit 1
    fi
else
    echo "Error: Nginx configuration test failed. Rolling back..." >&2
    cat /tmp/nginx_test_output.log >&2
    rm -f "$CONF_FILE_PATH"
    echo "[ROLLED BACK] Deleted ${CONF_FILE_PATH}" >&2
    exit 1
fi

rm -f /tmp/nginx_test_output.log

exit 0
