#!/usr/bin/env python3
"""
Fetch news headlines from major RSS feeds.
Zero dependencies — Python stdlib only (urllib + xml.etree).
"""

import argparse
import sys
import xml.etree.ElementTree as ET
from datetime import datetime
from email.utils import parsedate_to_datetime
from urllib.request import Request, urlopen
from urllib.error import URLError, HTTPError
from concurrent.futures import ThreadPoolExecutor, as_completed

FEEDS = {
    "bbc": {
        "top":      "https://feeds.bbci.co.uk/news/rss.xml",
        "world":    "https://feeds.bbci.co.uk/news/world/rss.xml",
        "business": "https://feeds.bbci.co.uk/news/business/rss.xml",
        "tech":     "https://feeds.bbci.co.uk/news/technology/rss.xml",
        "science":  "https://feeds.bbci.co.uk/news/science_and_environment/rss.xml",
        "health":   "https://feeds.bbci.co.uk/news/health/rss.xml",
    },
    "reuters": {
        "top":      "https://www.rss.reuters.com/news/topNews",
        "world":    "https://www.rss.reuters.com/news/worldNews",
        "business": "https://www.rss.reuters.com/news/businessNews",
        "tech":     "https://www.rss.reuters.com/news/technologyNews",
        "science":  "https://www.rss.reuters.com/news/scienceNews",
        "health":   "https://www.rss.reuters.com/news/healthNews",
    },
    "ap": {
        "top":      "https://rsshub.app/apnews/topics/apf-topnews",
    },
    "guardian": {
        "top":      "https://www.theguardian.com/international/rss",
        "world":    "https://www.theguardian.com/world/rss",
        "business": "https://www.theguardian.com/uk/business/rss",
        "tech":     "https://www.theguardian.com/uk/technology/rss",
        "science":  "https://www.theguardian.com/science/rss",
    },
    "aljazeera": {
        "top":      "https://www.aljazeera.com/xml/rss/all.xml",
    },
    "npr": {
        "top":      "https://feeds.npr.org/1001/rss.xml",
    },
    "dw": {
        "top":      "https://rss.dw.com/rss/en/top",
    },
}

UA = "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36"


def fetch_feed(source: str, category: str, url: str, limit: int, topic: str | None) -> list[dict]:
    """Fetch and parse a single RSS feed. Returns list of item dicts."""
    items = []
    try:
        req = Request(url, headers={"User-Agent": UA})
        with urlopen(req, timeout=15) as resp:
            data = resp.read()
        root = ET.fromstring(data)

        # Handle both RSS 2.0 and Atom
        ns = {"atom": "http://www.w3.org/2005/Atom"}
        rss_items = root.findall(".//item")
        if not rss_items:
            rss_items = root.findall(".//atom:entry", ns)

        for item in rss_items[:limit * 2]:  # fetch extra, filter later
            title = ""
            desc = ""
            link = ""
            pub_date = ""

            # RSS 2.0
            t = item.find("title")
            if t is not None and t.text:
                title = t.text.strip()
            d = item.find("description")
            if d is not None and d.text:
                desc = d.text.strip()
                # Strip HTML tags from description
                import re
                desc = re.sub(r"<[^>]+>", "", desc).strip()
                if len(desc) > 300:
                    desc = desc[:297] + "..."
            l = item.find("link")
            if l is not None and l.text:
                link = l.text.strip()
            p = item.find("pubDate")
            if p is not None and p.text:
                pub_date = p.text.strip()

            # Atom fallback
            if not title:
                t = item.find("atom:title", ns)
                if t is not None and t.text:
                    title = t.text.strip()
            if not link:
                l = item.find("atom:link", ns)
                if l is not None:
                    link = l.get("href", "")
            if not desc:
                d = item.find("atom:summary", ns)
                if d is not None and d.text:
                    import re
                    desc = re.sub(r"<[^>]+>", "", d.text.strip())
                    if len(desc) > 300:
                        desc = desc[:297] + "..."
            if not pub_date:
                p = item.find("atom:updated", ns)
                if p is not None and p.text:
                    pub_date = p.text.strip()

            if not title:
                continue

            # Topic filter
            if topic:
                topic_lower = topic.lower()
                if topic_lower not in title.lower() and topic_lower not in desc.lower():
                    continue

            # Parse date
            time_str = ""
            try:
                dt = parsedate_to_datetime(pub_date)
                time_str = dt.strftime("%Y-%m-%d %H:%M UTC")
            except Exception:
                time_str = pub_date[:25] if pub_date else ""

            items.append({
                "source": source,
                "category": category,
                "title": title,
                "description": desc,
                "link": link,
                "time": time_str,
            })

            if len(items) >= limit:
                break

    except (URLError, HTTPError, ET.ParseError) as e:
        items.append({
            "source": source,
            "category": category,
            "title": f"[Error fetching {source}/{category}: {e}]",
            "description": "",
            "link": "",
            "time": "",
        })
    return items


def format_output(all_items: list[dict]) -> str:
    """Format items as markdown grouped by source."""
    if not all_items:
        return "No news items found."

    grouped: dict[str, list[dict]] = {}
    for item in all_items:
        key = item["source"].upper()
        grouped.setdefault(key, []).append(item)

    lines = []
    for source, items in grouped.items():
        lines.append(f"## {source}\n")
        for it in items:
            cat = f" [{it['category']}]" if it["category"] != "top" else ""
            time = f" — {it['time']}" if it["time"] else ""
            lines.append(f"- **{it['title']}**{cat}{time}")
            if it["description"]:
                lines.append(f"  {it['description']}")
            if it["link"]:
                lines.append(f"  {it['link']}")
            lines.append("")
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Fetch news from RSS feeds")
    parser.add_argument("--source", "-s", help="Fetch from specific source only")
    parser.add_argument("--category", "-c", help="Fetch specific category (e.g. world, tech, business)")
    parser.add_argument("--topic", "-t", help="Filter headlines by keyword")
    parser.add_argument("--limit", "-n", type=int, default=8, help="Max items per feed (default: 8)")
    parser.add_argument("--list-sources", action="store_true", help="List available sources and categories")
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    args = parser.parse_args()

    if args.list_sources:
        for src, cats in sorted(FEEDS.items()):
            print(f"{src}: {', '.join(sorted(cats.keys()))}")
        return

    # Build list of (source, category, url) to fetch
    tasks = []
    if args.source:
        src = args.source.lower()
        if src not in FEEDS:
            print(f"Unknown source: {src}. Available: {', '.join(sorted(FEEDS.keys()))}", file=sys.stderr)
            sys.exit(1)
        cats = FEEDS[src]
        if args.category:
            if args.category not in cats:
                print(f"Unknown category '{args.category}' for {src}. Available: {', '.join(sorted(cats.keys()))}", file=sys.stderr)
                sys.exit(1)
            tasks.append((src, args.category, cats[args.category]))
        else:
            # Just fetch "top" for single source unless category specified
            if "top" in cats:
                tasks.append((src, "top", cats["top"]))
            else:
                first = next(iter(cats))
                tasks.append((src, first, cats[first]))
    else:
        # Fetch "top" from all sources
        for src, cats in FEEDS.items():
            if "top" in cats:
                tasks.append((src, "top", cats["top"]))

    # Fetch in parallel
    all_items = []
    with ThreadPoolExecutor(max_workers=8) as pool:
        futures = {
            pool.submit(fetch_feed, src, cat, url, args.limit, args.topic): (src, cat)
            for src, cat, url in tasks
        }
        for future in as_completed(futures):
            all_items.extend(future.result())

    # Sort by source name for consistent output
    source_order = list(FEEDS.keys())
    all_items.sort(key=lambda x: source_order.index(x["source"]) if x["source"] in source_order else 99)

    if args.json:
        import json
        print(json.dumps(all_items, indent=2, ensure_ascii=False))
    else:
        print(format_output(all_items))


if __name__ == "__main__":
    main()
