#!/usr/bin/env bash
set -euo pipefail

# Helper for monorepos: create/link/init a Netlify site for a specific subfolder.
#
# Usage:
#   ./netlify_monorepo_site.sh <site_dir> <netlify_site_name> [account_slug]
#
# Examples:
#   ./netlify_monorepo_site.sh sites/seattlecustomboatparts.com seattlecustomboatparts FuturePresentLabs
#
# Notes:
# - Runs netlify sites:create in <site_dir>.
# - Writes a Hugo netlify.toml (hugo --minify, publish public) if none exists.
# - Starts `netlify init` (interactive) to connect GitHub.

SITE_DIR="$1"
SITE_NAME="$2"
ACCOUNT_SLUG="${3:-}"

if [[ ! -d "$SITE_DIR" ]]; then
  echo "site_dir not found: $SITE_DIR" >&2
  exit 1
fi

pushd "$SITE_DIR" >/dev/null

if [[ ! -f netlify.toml ]]; then
  # Default Hugo version; override by editing file later if needed.
  "$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)/hugo_netlify_toml.sh" "0.155.1"
fi

CREATE_ARGS=(sites:create --name "$SITE_NAME" --with-ci)
if [[ -n "$ACCOUNT_SLUG" ]]; then
  CREATE_ARGS+=(--account-slug "$ACCOUNT_SLUG")
fi

echo "+ netlify ${CREATE_ARGS[*]}"
netlify "${CREATE_ARGS[@]}"

echo "+ netlify link (if prompted)"
netlify link || true

echo "+ netlify init (interactive: connect GitHub + set build settings)"
netlify init

popd >/dev/null

echo "Done. Next: commit netlify.toml (if new) and push; Netlify will deploy from GitHub."