# Example GCP Infrastructure Module

This is an example Terraform module generated by the `tf-module-generator` skill. It provisions a complete GCP infrastructure including VPC network, subnets, firewall rules, and Compute Engine instances.

## Architecture

```
Project (or dedicated)
└── VPC Network (default or custom)
    ├── Subnets (regional)
    │   ├── Compute Instances (with or without external IP)
    │   └── Cloud SQL (optional, if detected)
    └── Firewall Rules (allow SSH, HTTP, HTTPS)
```

## Features

- Multi-region subnet support
- Firewall rules for common services (SSH, HTTP, HTTPS)
- Configurable VM machine types and images
- Optional external IP assignment
- Comprehensive labeling
- IPv4 networking

## Requirements

| Name | Version |
|------|---------|
| terraform | >= 1.0.0 |
| google | ~> 5.0 |

## Providers

- `google` (HashiCorp) - Google Cloud Platform

## Usage

### Basic Example

```hcl
module "gcp_infrastructure" {
  source = "./example-gcp-module"

  gcp_region  = "us-central1"
  project     = "my-workspace-project"
  environment = "dev"

  network_name          = "network-example"
  vpc_cidr              = "10.0.0.0/16"
  private_subnet_cidrs  = ["10.0.1.0/24", "10.0.2.0/24"]
  public_subnet_cidrs   = ["10.0.101.0/24", "10.0.102.0/24"]

  instance_count       = 2
  machine_type         = "e2-micro"
  image_family         = "debian-12"
  admin_username       = "gcpuser"
  ssh_keys             = {
    gcpuser = "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAB..."
  }

  default_labels = {
    Owner       = "team-infrastructure"
    CostCenter  = "engineering"
    Environment = "dev"
  }
}
```

### Minimal Example

```hcl
module "minimal" {
  source = "./example-gcp-module"

  environment = "test"
}
```

This uses all default values and creates:
- 2 e2-micro Compute instances (Debian 12)
- VPC network with CIDR 10.0.0.0/16
- 2 public subnets and 2 private subnets (us-central1)
- Firewall rules allowing SSH (from anywhere - adjust!), HTTP, HTTPS
- Instances without external IPs (private) in private subnets

## Arguments

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| `gcp_region` | GCP region for all resources | `string` | `"us-central1"` | no |
| `project` | GCP project ID | `string` | `""` (uses provider default) | no |
| `environment` | Environment name for labeling | `string` | `"dev"` | no |
| `network_name` | Name for the VPC network | `string` | `"network-${var.environment}"` | no |
| `vpc_cidr` | CIDR block for the VPC network (auto-mode ignored) | `string` | `"10.0.0.0/16"` | no |
| `private_subnet_cidrs` | CIDR blocks for private subnets | `list(string)` | `["10.0.1.0/24", "10.0.2.0/24"]` | no |
| `public_subnet_cidrs` | CIDR blocks for public subnets | `list(string)` | `["10.0.101.0/24", "10.0.102.0/24"]` | no |
| `instance_count` | Number of Compute instances to create | `number` | `2` | no |
| `machine_type` | Machine type for instances | `string` | `"e2-micro"` | no |
| `image_family` | OS image family (e.g., debian-12, ubuntu-2204-lts) | `string` | `"debian-12"` | no |
| `admin_username` | Admin username for instances | `string` | `"gcpuser"` | no |
| `ssh_keys` | Map of username => SSH public key | `map(string)` | `{}` | no |
| `assign_external_ip` | Whether public subnets get external IPs | `bool` | `true` | no |
| `ssh_cidr_blocks` | CIDR blocks allowed to SSH | `list(string)` | `["0.0.0.0/0"]` | no |
| `default_labels` | Default labels for all resources | `map(string)` | see below | no |

### Default Labels

The module applies these labels to all resources that support labeling:

```hcl
{
  project     = "tf-module-generator-example"
  managed-by  = "Terraform"
  repo        = "https://github.com/example/repo"
  environment = var.environment
}
```

## Outputs

| Name | Description |
|------|-------------|
| `network_id` | ID of the VPC network |
| `network_name` | Name of the VPC network |
| `subnet_ids` | IDs of all subnets |
| `subnet_names` | Names of all subnets |
| `instance_ids` | IDs of the Compute instances |
| `instance_private_ips` | Private IP addresses of the instances |
| `instance_public_ips` | Public IP addresses (if assigned) |
| `firewall_rule_ids` | IDs of the firewall rules |

## Security Considerations

- **SSH Access**: By default, SSH is allowed from `0.0.0.0/0`. Change `ssh_cidr_blocks` to restrict access.
- **External IPs**: Instances in public subnets may have external IPs if `assign_external_ip = true`. This increases exposure. Use firewall rules to restrict inbound traffic.
- **OS Login**: Consider enabling OS Login for better SSH key management.
- **Service Accounts**: Assign minimal IAM roles to the service account used by instances.

## Cost Estimate (as of 2025, us-central1)

| Resource | Monthly Cost |
|----------|--------------|
| e2-micro instance (per instance) | ~$4.50 (if sustained use) |
| External IP (static) | ~$7.15 per IP |
| VPC Network | Free |
| Subnets | Free |
| Firewall Rules | Free |
| Cloud NAT (if used) | ~$23.04 per gateway + data |

**Total (2 instances + 2 external IPs)**: ~$22/month

## Clean Up

```bash
terraform destroy
```

## Generated By

This example was automatically generated by the `tf-module-generator` skill. Original infrastructure was discovered in an existing GCP project and converted to reusable Terraform code.

## License

MIT - see LICENSE for details.
