# Example Azure Infrastructure Module

This is an example Terraform module generated by the `tf-module-generator` skill. It provisions a complete Azure infrastructure including Resource Group, Virtual Network, subnets, NAT Gateway, Network Security Group, and Linux Virtual Machines.

## Architecture

```
Resource Group
└── Virtual Network (10.0.0.0/16)
    ├── Public Subnets (10.0.101.0/24, 10.0.102.0/24)
    │   └── NAT Gateway (outbound internet for private subnets)
    │   └── Network Security Group (SSH, HTTP, HTTPS)
    │   └── Linux Virtual Machines (optional public IP)
    └── Private Subnets (10.0.1.0/24, 10.0.2.0/24)
        └── Linux Virtual Machines (private IP only, NAT for outbound)
```

## Features

- Multi-region / zone support (using availability zones)
- Public and private subnets
- NAT Gateway for outbound internet access from private subnets
- Network Security Group with sensible defaults (SSH, HTTP, HTTPS)
- Configurable VM sizes and images
- Comprehensive tagging
- IPv4 networking

## Requirements

| Name | Version |
|------|---------|
| terraform | >= 1.0.0 |
| azurerm | ~> 4.0 |

## Providers

- `azurerm` (HashiCorp) - Microsoft Azure

## Usage

### Basic Example

```hcl
module "azure_infrastructure" {
  source = "./example-azure-module"

  azure_region  = "eastus"
  environment   = "dev"

  resource_group_name = "rg-example-dev"

  virtual_network_name = "vnet-example"
  vpc_cidr             = "10.0.0.0/16"

  private_subnet_cidrs = ["10.0.1.0/24", "10.0.2.0/24"]
  public_subnet_cidrs  = ["10.0.101.0/24", "10.0.102.0/24"]

  vm_size          = "Standard_B1s"
  admin_username   = "azureuser"
  ssh_public_key   = "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAB..."

  default_tags = {
    Owner       = "team-infrastructure"
    CostCenter  = "engineering"
    Environment = "dev"
  }
}
```

### Minimal Example

```hcl
module "minimal" {
  source = "./example-azure-module"

  environment = "test"
}
```

This uses all default values and creates:
- 2 Standard_B1s Linux VMs (Ubuntu 22.04)
- Resource Group "rg-example-test"
- VNet with CIDR 10.0.0.0/16
- 2 public subnets and 2 private subnets
- NAT Gateway and Public IP Prefix
- Network Security Group allowing SSH (from anywhere - adjust!), HTTP, HTTPS

## Arguments

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| `azure_region` | Azure region to deploy resources | `string` | `"eastus"` | no |
| `environment` | Environment name for tagging | `string` | `"dev"` | no |
| `resource_group_name` | Name for the resource group | `string` | `"rg-${var.environment}"` | no |
| `virtual_network_name` | Name for the virtual network | `string` | `"vnet-${var.environment}"` | no |
| `vpc_cidr` | CIDR block for the VNet | `string` | `"10.0.0.0/16"` | no |
| `private_subnet_cidrs` | CIDR blocks for private subnets | `list(string)` | `["10.0.1.0/24", "10.0.2.0/24"]` | no |
| `public_subnet_cidrs` | CIDR blocks for public subnets | `list(string)` | `["10.0.101.0/24", "10.0.102.0/24"]` | no |
| `vm_size` | Azure VM size | `string` | `"Standard_B1s"` | no |
| `admin_username` | Admin username for VMs | `string` | `"azureuser"` | no |
| `ssh_public_key` | SSH public key content (use `file()` to read) | `string` | `""` | no |
| `ssh_cidr_blocks` | CIDR blocks allowed to SSH | `list(string)` | `["0.0.0.0/0"]` | no |
| `default_tags` | Default tags for all resources | `map(string)` | see below | no |

### Default Tags

The module applies these tags to all resources:

```hcl
{
  Project     = "tf-module-generator-example"
  ManagedBy   = "Terraform"
  Repo        = "https://github.com/example/repo"
  Environment = var.environment
}
```

## Outputs

| Name | Description |
|------|-------------|
| `resource_group_id` | ID of the resource group |
| `vnet_id` | ID of the virtual network |
| `public_subnet_ids` | IDs of the public subnets |
| `private_subnet_ids` | IDs of the private subnets |
| `vm_ids` | IDs of the virtual machines |
| `vm_private_ips` | Private IP addresses of VMs |
| `vm_public_ips` | Public IP addresses of VMs (if assigned) |
| `nsg_id` | ID of the network security group |
| `nat_gateway_id` | ID of the NAT Gateway |

## Security Considerations

- **SSH Access**: By default, SSH is allowed from `0.0.0.0/0`. Change `ssh_cidr_blocks` to restrict access.
- **NAT Gateway**: NAT Gateway incurs hourly + data processing charges. Consider using only public subnets for cost-sensitive environments.
- **Public IPs**: VMs in public subnets get public IPs; this increases exposure. Use NSG rules to restrict inbound traffic.
- **SSH Key**: Using SSH keys is more secure than passwords. Ensure `ssh_public_key` is set and `admin_password` is null.

## Cost Estimate (as of 2025, East US)

| Resource | Monthly Cost |
|----------|--------------|
| Linux VM (B1s) | ~$8.50 per instance |
| NAT Gateway | ~$33.60 |
| Public IP (static) | ~$4.48 each |
| VNet | Free |
| Subnets | Free |
| NSG | Free |
| Resource Group | Free |

**Total (2 VMs + NAT + 2 Public IPs)**: ~$60/month

## Clean Up

```bash
terraform destroy
```

## Generated By

This example was automatically generated by the `tf-module-generator` skill. Original infrastructure was discovered in an existing Azure account and converted to reusable Terraform code.

## License

MIT - see LICENSE for details.
