# Example AWS Infrastructure Module

This is an example Terraform module generated by the `tf-module-generator` skill. It provisions a complete AWS infrastructure including VPC, subnets, NAT gateways, security groups, and EC2 instances.

## Architecture

```
VPC (10.0.0.0/16)
├── Public Subnets (10.0.101.0/24, 10.0.102.0/24)
│   └── NAT Gateways (one per AZ)
│   └── EC2 Instances (optional public IP)
└── Private Subnets (10.0.1.0/24, 10.0.2.0/24)
    └── EC2 Instances (private IP only)
```

## Features

- Multi-AZ deployment for high availability
- Public and private subnets
- NAT Gateways for outbound internet access from private subnets
- Internet Gateway for public internet access
- Security groups with sensible defaults (SSH, HTTP, HTTPS)
- Configurable instance types and AMIs
- Comprehensive tagging
- IPv4-only networking

## Requirements

| Name | Version |
|------|---------|
| terraform | >= 1.0.0 |
| aws | ~> 5.0 |

## Providers

- `aws` (HashiCorp) - Amazon Web Services

## Usage

### Basic Example

```hcl
module "aws_infrastructure" {
  source = "./example-aws-module"

  aws_region  = "us-east-1"
  environment = "dev"

  vpc_cidr           = "10.0.0.0/16"
  availability_zones = ["us-east-1a", "us-east-1b"]

  private_subnet_cidrs = ["10.0.1.0/24", "10.0.2.0/24"]
  public_subnet_cidrs  = ["10.0.101.0/24", "10.0.102.0/24"]

  instance_type = "t3.micro"
  ami_id        = "ami-0c55b159cbfafe1f0"  # Amazon Linux 2023 (us-east-1)
  key_name      = "my-ssh-key"

  default_tags = {
    Owner       = "team-infrastructure"
    CostCenter  = "engineering"
    Environment = "dev"
  }
}
```

### Minimal Example

```hcl
module "minimal" {
  source = "./example-aws-module"

  environment = "test"
}
```

This uses all default values and creates:
- 2 t3.micro EC2 instances (Amazon Linux 2023 AMI)
- VPC with CIDR 10.0.0.0/16
- 2 public subnets and 2 private subnets
- NAT Gateways and Internet Gateway
- Security groups allowing SSH (from anywhere - adjust!), HTTP, HTTPS

## Arguments

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| `aws_region` | AWS region to deploy resources | `string` | `"us-east-1"` | no |
| `environment` | Environment name for tagging | `string` | `"dev"` | no |
| `vpc_cidr` | CIDR block for the VPC | `string` | `"10.0.0.0/16"` | no |
| `availability_zones` | List of availability zones | `list(string)` | `["us-east-1a", "us-east-1b"]` | no |
| `private_subnet_cidrs` | CIDR blocks for private subnets | `list(string)` | `["10.0.1.0/24", "10.0.2.0/24"]` | no |
| `public_subnet_cidrs` | CIDR blocks for public subnets | `list(string)` | `["10.0.101.0/24", "10.0.102.0/24"]` | no |
| `instance_type` | EC2 instance type | `string` | `"t3.micro"` | no |
| `ami_id` | AMI ID for EC2 instances | `string` | (Amazon Linux 2023 in us-east-1) | no |
| `key_name` | SSH key pair name | `string` | `null` | no |
| `ssh_cidr_blocks` | CIDR blocks allowed to SSH | `list(string)` | `["0.0.0.0/0"]` | no |
| `default_tags` | Default tags for all resources | `map(string)` | see below | no |

### Default Tags

The module applies these tags to all resources:

```hcl
{
  Project     = "tf-module-generator-example"
  ManagedBy   = "Terraform"
  Repo        = "https://github.com/example/repo"
  Environment = var.environment
}
```

## Outputs

| Name | Description |
|------|-------------|
| `vpc_id` | ID of the created VPC |
| `vpc_cidr_block` | CIDR block of the VPC |
| `public_subnet_ids` | IDs of the public subnets |
| `private_subnet_ids` | IDs of the private subnets |
| `instance_ids` | IDs of the EC2 instances |
| `instance_public_ips` | Public IP addresses of EC2 instances |
| `instance_private_ips` | Private IP addresses of EC2 instances |
| `security_group_id` | ID of the instance security group |

## Security Considerations

- **SSH Access**: By default, SSH is allowed from `0.0.0.0/0`. Change `ssh_cidr_blocks` to restrict access.
- **NAT Gateways**: NAT Gateways incur hourly charges. Consider using only public subnets for cost-sensitive environments.
- **Instance Key Pair**: Using SSH key pairs is more secure than passwords. Ensure `key_name` points to an existing key pair.
- **AMI Selection**: The default AMI is specific to `us-east-1`. Override `ami_id` for other regions.

## Cost Estimate (as of 2025)

| Resource | Monthly Cost (us-east-1) |
|----------|-------------------------|
| EC2 (t3.micro, Linux) | ~$8.50 per instance |
| NAT Gateway | ~$33.60 per AZ |
| EIP (attached) | Free |
| EIP (idle) | ~$3.60 each |
| VPC | Free |
| Subnets | Free |
| Security Groups | Free |

**Total (2 instances + 2 NAT Gateways)**: ~$74/month

## Clean Up

```bash
terraform destroy
```

## Generated By

This example was automatically generated by the `tf-module-generator` skill. Original infrastructure was discovered in an existing AWS account and converted to reusable Terraform code.

## License

MIT - see LICENSE for details.
