#!/usr/bin/env python3
"""
Test template generator for py-test-creator.

This module generates pytest-compatible test code from parsed function
information. It creates test functions with appropriate assertions and
docstrings, handling various parameter types and edge cases.
"""

from typing import List, Dict, Any, Optional
from dataclasses import dataclass, field
from scripts.parser import Function, Parameter


@dataclass
class TestConfig:
    """Configuration for test generation."""
    module_name: str
    coverage: bool = False
    strict: bool = False


class TestGenerator:
    """
    Generates pytest test code from function definitions.

    Takes a list of parsed Function objects and produces a complete
    pytest test file with test methods covering various scenarios.
    """

    def __init__(self, functions: List[Function], module_name: str, coverage: bool = False, strict: bool = False):
        """
        Initialize the test generator.

        Args:
            functions: List of Function objects to generate tests for
            module_name: Name of the module being tested (used in imports)
            coverage: Whether to include coverage hints
            strict: Enable strict mode for additional validation
        """
        self.functions = functions
        self.config = TestConfig(module_name=module_name, coverage=coverage, strict=strict)
        self.output_lines: List[str] = []

    def generate(self) -> str:
        """
        Generate the complete test file content.

        Returns:
            String containing the generated test code
        """
        self._write_header()
        self._write_imports()
        self._write_fixtures()
        self._write_tests()
        return "\n".join(self.output_lines)

    def _write_header(self):
        """Write file header with docstring and encoding."""
        self.output_lines.append('"""')
        self.output_lines.append(f"Automatically generated pytest tests for {self.config.module_name}")
        self.output_lines.append("")
        self.output_lines.append("This file was generated by py-test-creator. DO NOT EDIT MANUALLY.")
        self.output_lines.append('"""')
        self.output_lines.append("")
        self.output_lines.append("# flake8: noqa")
        self.output_lines.append("# type: ignore")

        if self.config.coverage:
            self.output_lines.append("# pragma: no cover")
        self.output_lines.append("")

    def _write_imports(self):
        """Write necessary import statements."""
        self.output_lines.append("import pytest")
        self.output_lines.append(f"from {self.config.module_name} import *")
        self.output_lines.append("")

    def _write_fixtures(self):
        """Write pytest fixtures if needed."""
        # Only include fixtures if we detect a need for them
        # For now, we'll keep it simple and not add fixtures
        self.output_lines.append("")

    def _write_tests(self):
        """Write test functions for all parsed functions."""
        for func in self.functions:
            self._write_test_function(func)
            self.output_lines.append("")

    def _write_test_function(self, func: Function):
        """
        Write test function(s) for a single function.

        Args:
            func: Function object to generate tests for
        """
        # Determine test name prefix
        if func.is_method and func.class_name:
            test_prefix = f"test_{func.class_name.lower()}_{func.name}"
        else:
            test_prefix = f"test_{func.name}"

        # Write basic positive test
        self._write_basic_test(func, test_prefix)

        # Write edge case tests based on parameter types
        self._write_edge_case_tests(func, test_prefix)

        # Write tests for required parameters
        if func.get_required_params():
            self._write_required_param_tests(func, test_prefix)

        # Write tests for optional parameters
        if func.get_optional_params():
            self._write_optional_param_tests(func, test_prefix)

        # Write tests for special parameters (*args, **kwargs)
        if func.has_var_positional() or func.has_var_keyword():
            self._write_varargs_tests(func, test_prefix)

        # Write exception test if docstring indicates error conditions
        if func.docstring and ("raise" in func.docstring.lower() or "error" in func.docstring.lower()):
            self._write_exception_test(func, test_prefix)

    def _write_basic_test(self, func: Function, test_prefix: str):
        """
        Write a basic positive test with example values.

        Args:
            func: Function being tested
            test_prefix: Prefix for test function names
        """
        test_name = f"{test_prefix}_basic"
        docstring = f"Test {func.name} with basic valid arguments."

        # Generate example arguments
        args_str = self._generate_example_args(func.parameters)

        # Generate expected result (placeholder)
        if func.returns:
            result_comment = f"# Expected: {func.returns}"
        else:
            result_comment = ""

        self._write_test_function_start(test_name, docstring)
        self.output_lines.append(f"    result = {func.name}({args_str})")
        if result_comment:
            self.output_lines.append(f"    {result_comment}")
        self.output_lines.append("    assert result is not None")
        self.output_lines.append("")

    def _write_required_param_tests(self, func: Function, test_prefix: str):
        """Write tests focusing on required parameters."""
        required = func.get_required_params()

        if required:
            test_name = f"{test_prefix}_required_params"
            docstring = f"Test {func.name} with required parameters only."

            self._write_test_function_start(test_name, docstring)

            # Build args with just required params
            args = []
            for param in required:
                example = self._get_type_example(param)
                args.append(f"{param.name}={example}")

            args_str = ", ".join(args)
            self.output_lines.append(f"    result = {func.name}({args_str})")
            self.output_lines.append("    assert result is not None")
            self.output_lines.append("")

    def _write_optional_param_tests(self, func: Function, test_prefix: str):
        """Write tests for optional parameters."""
        optional = func.get_optional_params()

        for param in optional:
            test_name = f"{test_prefix}_optional_{param.name}"
            docstring = f"Test {func.name} with optional parameter {param.name}."

            self._write_test_function_start(test_name, docstring)

            # Build args: all required + this optional
            all_params = func.get_required_params() + [param]
            args_list = []
            for p in all_params:
                example = self._get_type_example(p)
                args_list.append(f"{p.name}={example}")

            args_str = ", ".join(args_list)
            self.output_lines.append(f"    result = {func.name}({args_str})")
            self.output_lines.append("    assert result is not None")
            self.output_lines.append("")

    def _write_edge_case_tests(self, func: Function, test_prefix: str):
        """Write edge case tests based on parameter types."""
        for param in func.parameters:
            if param.annotation:
                test_name = f"{test_prefix}_{param.name}_edge_cases"
                docstring = f"Test {func.name} with edge case values for {param.name}."

                self._write_test_function_start(test_name, docstring)
                self.output_lines.append(f"    # Test with various valid/invalid values for {param.annotation}")
                self.output_lines.append(f"    # TODO: Add specific edge cases based on type")
                self.output_lines.append("    pytest.skip('Edge case test needs customization')")
                self.output_lines.append("")

    def _write_varargs_tests(self, func: Function, test_prefix: str):
        """Write tests for *args and **kwargs."""
        if func.has_var_positional():
            test_name = f"{test_prefix}_varargs"
            docstring = f"Test {func.name} with variable positional arguments (*args)."

            self._write_test_function_start(test_name, docstring)
            self.output_lines.append("    # Test with multiple additional arguments")
            self.output_lines.append("    pytest.skip('Varargs test needs customization based on function signature')")
            self.output_lines.append("")

        if func.has_var_keyword():
            test_name = f"{test_prefix}_kwargs"
            docstring = f"Test {func.name} with variable keyword arguments (**kwargs)."

            self._write_test_function_start(test_name, docstring)
            self.output_lines.append("    # Test with additional keyword arguments")
            self.output_lines.append("    pytest.skip('Kwargs test needs customization based on function signature')")
            self.output_lines.append("")

    def _write_exception_test(self, func: Function, test_prefix: str):
        """Write a test expecting an exception."""
        test_name = f"{test_prefix}_raises"
        docstring = f"Test {func.name} raises appropriate exceptions for invalid input."

        self._write_test_function_start(test_name, docstring)
        self.output_lines.append("    # Based on docstring, this function should raise exceptions")
        self.output_lines.append("    with pytest.raises(Exception):")
        self.output_lines.append("        # TODO: Provide invalid arguments that trigger the exception")
        self.output_lines.append("        pass")
        self.output_lines.append("")

    def _write_test_function_start(self, name: str, docstring: str):
        """Write the beginning of a test function."""
        self.output_lines.append(f"def {name}():")
        if docstring:
            self.output_lines.append(f'    """{docstring}"""')
        else:
            self.output_lines.append("    pass")

    def _generate_example_args(self, parameters: List[Parameter]) -> str:
        """
        Generate example argument string for a function call.

        Args:
            parameters: List of parameters

        Returns:
            Comma-separated string of keyword arguments
        """
        args = []
        for param in parameters:
            example = self._get_type_example(param)
            args.append(f"{param.name}={example}")
        return ", ".join(args)

    def _get_type_example(self, param: Parameter) -> str:
        """
        Get an example value for a parameter based on its type annotation.

        Args:
            param: Parameter object

        Returns:
            String representation of an example value
        """
        annotation = param.annotation if param.annotation else "Any"

        # Map type hints to example values
        type_examples = {
            "int": "42",
            "float": "3.14",
            "str": '"example"',
            "bool": "True",
            "list": "[]",
            "dict": "{}",
            "tuple": "()",
            "set": "set()",
            "None": "None",
            "Any": "None",
            "bytes": "b'bytes'",
        }

        # Handle generic types like List[int], Dict[str, int], etc.
        if "[" in annotation:
            base_type = annotation.split("[")[0]
            return type_examples.get(base_type, "None")

        return type_examples.get(annotation, "None")

    def _format_value(self, value: Any) -> str:
        """Format a Python value as a string literal."""
        if isinstance(value, str):
            return repr(value)
        elif isinstance(value, bool):
            return str(value)
        elif value is None:
            return "None"
        else:
            return repr(value)
