#!/usr/bin/env node
/**
 * GraphQL TypeScript Generator - Type Generator
 * Core logic for transforming GraphQL schema to TypeScript types
 */

import {
  GraphQLSchema,
  GraphQLObjectType,
  GraphQLInterfaceType,
  GraphQLUnionType,
  GraphQLEnumType,
  GraphQLInputObjectType,
  GraphQLArgument,
  GraphQLList,
  GraphQLNonNull
} from 'graphql';
import { formatDescription, isBuiltInScalar } from './utils.js';
import { join } from 'path';

export interface GeneratorOptions {
  namespace?: string;
  prefix?: string;
  skipEnums?: boolean;
  skipUnions?: boolean;
  schemaName?: string;
}

export interface GeneratedFile {
  filename: string;
  content: string;
}

/**
 * Main generator class
 */
export class TypeGenerator {
  private schema: GraphQLSchema;
  private options: GeneratorOptions;
  private generatedTypes: Set<string> = new Set();
  private outputLines: string[] = [];

  constructor(schema: GraphQLSchema, options: GeneratorOptions = {}) {
    this.schema = schema;
    this.options = {
      skipEnums: false,
      skipUnions: false,
      ...options
    };
  }

  /**
   * Generate all TypeScript types from schema
   */
  generate(): GeneratedFile[] {
    const files: GeneratedFile[] = [];

    // Generate main types file
    this.outputLines = [];
    this.addHeader();

    // Generate enums first (they're referenced by other types)
    if (!this.options.skipEnums) {
      this.generateEnums();
    }

    // Generate interfaces for object types
    this.generateObjectTypes();

    // Generate input types
    this.generateInputTypes();

    // Generate unions
    if (!this.options.skipUnions) {
      this.generateUnions();
    }

    // Generate additional helper types (like query/mutation types)
    this.generateOperationTypes();

    files.push({
      filename: this.getOutputFilename(),
      content: this.outputLines.join('\n')
    });

    return files;
  }

  /**
   * Add file header with imports and comments
   */
  private addHeader(): void {
    const header = `/**
 * Auto-generated TypeScript types from GraphQL schema
 * Generated by graphql-ts-generator
 * caution: Manual edits may be overwritten
 */

export {};
`;
    this.outputLines.push(header);
  }

  /**
   * Generate enum types
   */
  private generateEnums(): void {
    const typeMap = this.schema.getTypeMap();

    for (const typeName in typeMap) {
      const type = typeMap[typeName];

      if (type instanceof GraphQLEnumType && !typeName.startsWith('__')) {
        this.generateEnum(type);
      }
    }
  }

  /**
   * Generate a single enum
   */
  private generateEnum(type: GraphQLEnumType): void {
    const enumName = this.options.prefix ? `${this.options.prefix}${type.name}` : type.name;

    this.outputLines.push(formatDescription(type.description));
    this.outputLines.push(`export enum ${enumName} {`);

    type.getValues().forEach(value => {
      const valueName = value.name;
      const valueLiteral = value.value !== undefined ? ` = ${JSON.stringify(value.value)}` : '';
      this.outputLines.push(`  ${valueName}${valueLiteral},`);
    });

    this.outputLines.push(`}\n`);
    this.generatedTypes.add(enumName);
  }

  /**
   * Generate object and interface types
   */
  private generateObjectTypes(): void {
    const typeMap = this.schema.getTypeMap();

    for (const typeName in typeMap) {
      const type = typeMap[typeName];

      // Skip enums, unions, and internal types
      if (
        type instanceof GraphQLEnumType ||
        type instanceof GraphQLUnionType ||
        typeName.startsWith('__')
      ) {
        continue;
      }

      if (type instanceof GraphQLObjectType || type instanceof GraphQLInterfaceType) {
        this.generateObjectType(type);
      }
    }
  }

  /**
   * Generate a single object or interface type
   */
  private generateObjectType(type: GraphQLObjectType | GraphQLInterfaceType): void {
    const isInterface = type instanceof GraphQLInterfaceType;
    const typeName = this.options.prefix ? `${this.options.prefix}${type.name}` : type.name;
    const keyword = 'interface'; // Always use interface for flexibility

    this.outputLines.push(formatDescription(type.description));
    this.outputLines.push(`export ${keyword} ${typeName} {`);

    const fields = type.getFields();

    for (const fieldName in fields) {
      const field = fields[fieldName];
      const fieldType = this.typeToString(field.type);
      const description = formatDescription(field.description);

      if (description) {
        this.outputLines.push(`  ${description}`);
      }

      this.outputLines.push(`  ${fieldName}: ${fieldType};`);
    }

    this.outputLines.push(`}\n`);
    this.generatedTypes.add(typeName);
  }

  /**
   * Convert GraphQL type to TypeScript type string
   * Handles List, NonNull, scalars, enums, objects, unions, interfaces
   */
  private typeToString(graphqlType: any): string {
    let isList = false;
    let isNonNull = false;
    let current: any = graphqlType;

    // Unwrap wrappers
    while (true) {
      if (current instanceof GraphQLList) {
        isList = true;
        current = current.ofType;
      } else if (current instanceof GraphQLNonNull) {
        isNonNull = true;
        current = current.ofType;
      } else {
        break;
      }
    }

    const typeName = current.name;
    let tsType: string;

    // Check if it's a built-in scalar
    if (isBuiltInScalar(typeName)) {
      const scalarMap: Record<string, string> = {
        'Int': 'number',
        'Float': 'number',
        'String': 'string',
        'Boolean': 'boolean',
        'ID': 'string'
      };
      tsType = scalarMap[typeName] || 'any';
    } else {
      // It's a named type (object, interface, enum, union, or input)
      tsType = this.options.prefix ? `${this.options.prefix}${typeName}` : typeName;
    }

    // Apply list array type
    if (isList) {
      tsType += '[]';
    }

    // Add null if not non-null
    if (!isNonNull) {
      tsType += ' | null';
    }

    return tsType;
  }

  /**
   * Generate input object types
   */
  private generateInputTypes(): void {
    const typeMap = this.schema.getTypeMap();

    for (const typeName in typeMap) {
      const type = typeMap[typeName];

      if (type instanceof GraphQLInputObjectType && !typeName.startsWith('__')) {
        this.generateInputType(type);
      }
    }
  }

  /**
   * Generate a single input type
   */
  private generateInputType(type: GraphQLInputObjectType): void {
    const typeName = this.options.prefix ? `${this.options.prefix}${type.name}` : type.name;

    this.outputLines.push(formatDescription(type.description));
    this.outputLines.push(`export interface ${typeName} {`);

    const fields = type.getFields();

    for (const fieldName in fields) {
      const field = fields[fieldName];
      const fieldType = this.typeToString(field.type);
      const description = formatDescription(field.description);

      if (description) {
        this.outputLines.push(`  ${description}`);
      }

      // If field has a default value, it's optional in TS
      const isOptional = field.defaultValue !== undefined;
      this.outputLines.push(`  ${fieldName}${isOptional ? '?' : ''}: ${fieldType};`);
    }

    this.outputLines.push(`}\n`);
    this.generatedTypes.add(typeName);
  }

  /**
   * Generate union types
   */
  private generateUnions(): void {
    const typeMap = this.schema.getTypeMap();

    for (const typeName in typeMap) {
      const type = typeMap[typeName];

      if (type instanceof GraphQLUnionType && !typeName.startsWith('__')) {
        this.generateUnion(type);
      }
    }
  }

  /**
   * Generate a single union type
   */
  private generateUnion(type: GraphQLUnionType): void {
    const unionName = this.options.prefix ? `${this.options.prefix}${type.name}` : type.name;
    const memberTypes = type.getTypes().map(t => {
      const prefix = this.options.prefix || '';
      return `${prefix}${t.name}`;
    });

    this.outputLines.push(formatDescription(type.description));
    this.outputLines.push(`export type ${unionName} = ${memberTypes.join(' | ')};\n`);
    this.generatedTypes.add(unionName);
  }

  /**
   * Generate Query, Mutation, and Subscription operation types
   */
  private generateOperationTypes(): void {
    const queryType = this.schema.getQueryType();
    const mutationType = this.schema.getMutationType();
    const subscriptionType = this.schema.getSubscriptionType();

    if (queryType) {
      this.generateOperationInterface('Query', queryType);
    }

    if (mutationType) {
      this.generateOperationInterface('Mutation', mutationType);
    }

    if (subscriptionType) {
      this.generateOperationInterface('Subscription', subscriptionType);
    }
  }

  /**
   * Generate operation interface (Query/Mutation/Subscription)
   */
  private generateOperationInterface(operationName: string, type: GraphQLObjectType): void {
    const prefixedName = this.options.prefix ? `${this.options.prefix}${operationName}` : operationName;

    this.outputLines.push(`/** Auto-generated ${operationName} operation interface */`);
    this.outputLines.push(`export interface ${prefixedName} {`);

    const fields = type.getFields();
    for (const fieldName in fields) {
      const field = fields[fieldName];
      const fieldType = this.typeToString(field.type);
      const args = this.formatArguments(field.args);
      const argsString = args.length > 0 ? `(${args})` : '';
      this.outputLines.push(`  ${fieldName}${argsString}: ${fieldType};`);
    }

    this.outputLines.push(`}\n`);
  }

  /**
   * Format GraphQL arguments as TypeScript parameters
   */
  private formatArguments(args: Record<string, GraphQLArgument>): string {
    const argList: string[] = [];

    for (const argName in args) {
      const arg = args[argName];
      const argType = this.typeToString(arg.type);
      // For input object arguments, if the input field is required (non-null), we don't add '?'
      // But we need to check if it's nullable at the outer level
      const isOptional = !(arg.type instanceof GraphQLNonNull);
      argList.push(`${argName}${isOptional ? '?' : ''}: ${argType}`);
    }

    return argList;
  }

  /**
   * Get output filename
   */
  private getOutputFilename(): string {
    const schemaName = this.options.schemaName || 'types';
    const prefix = this.options.prefix ? `${this.options.prefix}_` : '';
    return `${prefix}${schemaName}.generated.ts`;
  }
}

/**
 * Get output directory for generated files
 */
export function getOutputDir(outputPath: string): string {
  try {
    // Create relative path if not absolute
    if (!outputPath.startsWith('/') && !outputPath.match(/^[a-zA-Z]:\\/)) {
      return join(process.cwd(), outputPath);
    }
    return outputPath;
  } catch {
    return join(process.cwd(), 'dist');
  }
}
