#!/usr/bin/env node

const https = require('https');
const fs = require('fs').promises;
const path = require('path');
const os = require('os');

const CONFIG_DIR = path.join(os.homedir(), '.near-email');
const CONFIG_FILE = path.join(CONFIG_DIR, 'config.json');

/**
 * Load email configuration
 */
async function loadConfig() {
  try {
    const data = await fs.readFile(CONFIG_FILE, 'utf8');
    return JSON.parse(data);
  } catch (error) {
    return null;
  }
}

/**
 * Save email configuration
 */
async function saveConfig(config) {
  try {
    await fs.mkdir(CONFIG_DIR, { recursive: true });
    await fs.writeFile(CONFIG_FILE, JSON.stringify(config, null, 2), { mode: 0o600 });
    return true;
  } catch (error) {
    throw new Error(`Failed to save config: ${error.message}`);
  }
}

/**
 * Query NEAR RPC for account info
 */
async function queryAccount(accountId) {
  return new Promise((resolve, reject) => {
    const data = JSON.stringify({
      jsonrpc: '2.0',
      id: 'dontcare',
      method: 'query',
      params: {
        request_type: 'view_account',
        finality: 'final',
        account_id: accountId,
      },
    });

    const options = {
      hostname: 'rpc.mainnet.near.org',
      port: 443,
      path: '/',
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Content-Length': data.length,
      },
    };

    const req = https.request(options, (res) => {
      let body = '';
      res.on('data', (chunk) => body += chunk);
      res.on('end', () => {
        try {
          resolve(JSON.parse(body));
        } catch (e) {
          reject(new Error(`Failed to parse response: ${e.message}`));
        }
      });
    });

    req.on('error', reject);
    req.write(data);
    req.end();
  });
}

/**
 * Format NEAR amount
 */
function formatNear(amount) {
  const yoctoToNear = 1e24;
  const near = Number(BigInt(amount)) / yoctoToNear;
  return near.toFixed(4);
}

/**
 * Generate email report
 */
function generateReport(accountId, balance, transactions) {
  const date = new Date().toLocaleDateString();

  return `
NEAR Account Report
====================
Date: ${date}
Account: ${accountId}

Balance: ${balance} NEAR

Recent Transactions:
${transactions.length > 0 ? transactions.map(t =>
  `- ${t.type}: ${formatNear(t.amount)} NEAR (${t.date})`
).join('\n') : 'No recent transactions'}

---
Generated by NEAR Email Reporter
  `.trim();
}

// CLI interface
const command = process.argv[2];
const args = process.argv.slice(3);

async function main() {
  try {
    switch (command) {
      case 'setup': {
        const options = {};
        for (let i = 0; i < args.length; i++) {
          if (args[i] === '--host') options.host = args[++i];
          else if (args[i] === '--port') options.port = parseInt(args[++i]);
          else if (args[i] === '--user') options.user = args[++i];
          else if (args[i] === '--pass') options.pass = args[++i];
          else if (args[i] === '--from') options.from = args[++i];
          else if (args[i] === '--secure') options.secure = true;
        }

        if (!options.host || !options.user || !options.pass || !options.from) {
          console.error('Error: Missing required options');
          console.error('Required: --host, --user, --pass, --from');
          console.error('Optional: --port, --secure');
          process.exit(1);
        }

        options.port = options.port || 587;
        await saveConfig(options);
        console.log('✅ Email configuration saved!');
        break;
      }

      case 'report': {
        const accountId = args[0];
        const recipient = args[1];

        if (!accountId) {
          console.error('Error: Account ID required');
          console.error('Usage: near-email report <account_id> [recipient]');
          process.exit(1);
        }

        const config = await loadConfig();
        if (!config) {
          console.error('Error: Email not configured. Run near-email setup first.');
          process.exit(1);
        }

        const result = await queryAccount(accountId);
        if (result.result) {
          const balance = formatNear(result.result.amount);
          const report = generateReport(accountId, balance, []);

          console.log(report);
          console.log('\n(Note: Email sending requires nodemailer installation)');
        }
        break;
      }

      case 'alert':
        console.log('Alert functionality requires nodemailer for email sending.');
        console.log('Install: npm install nodemailer');
        console.log(`Would set up alert for ${args[0]} at ${args[1]} NEAR threshold`);
        break;

      case 'schedule':
        console.log('Schedule functionality requires cron or job scheduler.');
        console.log(`Would schedule reports for ${args[0]} with cron: ${args[1]}`);
        break;

      default:
        console.log('NEAR Email Reporter');
        console.log('');
        console.log('Commands:');
        console.log('  setup --host <h> --port <p> --user <u> --pass <p> --from <e>');
        console.log('  report <account> [recipient]   Send transaction report');
        console.log('  alert <account> <threshold>   Set up balance alert');
        console.log('  schedule <account> <cron>     Schedule periodic reports');
    }
  } catch (error) {
    console.error(`Error: ${error.message}`);
    process.exit(1);
  }
}

main();
