---
name: web-claude
description: Unified web search skill. Fallback order — web_search(Brave) → duckduckgo → claude.ai. Auto-cache search results (saved to memory/research/)
author: 무펭이 🐧
---

# Unified Web Search 🐧

Reliable web search via 3-tier fallback strategy: **Brave API → DuckDuckGo → claude.ai browser**

## Search Strategy

### Tier 1: web_search (Brave API) — ⚡ Fast and Reliable (Recommended)

Use OpenClaw built-in `web_search` tool.

```
web_search(query="search query", count=5, freshness="pw")
```

**Pros:**
- Fast response (1-2s)
- Structured JSON results
- freshness parameter support (pd=24h, pw=1 week, pm=1 month)
- search_lang, country parameters for Korean/regional search

**Cons:**
- Requires Brave API key
- `missing_brave_api_key` error without key

### Tier 2: duckduckgo-search — 🔒 Privacy-focused (Fallback)

Use DuckDuckGo API when Brave fails.

```bash
python -c "
from duckduckgo_search import DDGS

with DDGS() as ddgs:
    results = list(ddgs.text('query', region='wt-wt', max_results=5))
    for r in results:
        print(f\"{r['title']}: {r['href']}\")
"
```

**Pros:**
- No API key required
- Privacy-friendly
- Various search types (text, news, images, videos)

**Cons:**
- Lower result quality than Brave
- Request limits (blocks if too many consecutive requests)

### Tier 3: web-claude (Browser) — 🧠 Analysis+Search (Last Resort)

Trigger web search in claude.ai browser tab.

```
1. browser navigate → https://claude.ai/new
2. browser act type → "search question"
3. browser act press → Enter
4. sleep 15-30s
5. browser snapshot → extract response
```

**Pros:**
- No API key required
- claude.ai auto web search + analyze + summarize
- Useful for complex research

**Cons:**
- Slow (15-30s)
- Requires browser (port 18800, openclaw profile)
- Daily message limit (free account)
- Automation detection risk

## Auto-fallback Logic

```
if web_search available:
    try web_search(query)
    if success: return results
    
if web_search failed or unavailable:
    try duckduckgo-search
    if success: return results
    
if both failed:
    fallback to web-claude browser method
```

## Search Results Auto-cache

All search results automatically saved to `memory/research/` folder:

**Filename Convention:**
```
memory/research/search-YYYY-MM-DD-HH-MM-[keyword].md
```

**Saved Content:**
- Search query
- Timestamp
- Search method used (Brave/DuckDuckGo/claude.ai)
- Search results (links + summaries)
- Extracted insights

**Example:**
```markdown
# Web Search: AI Agent Market Size

- **Search Time:** 2026-02-14 07:56 KST
- **Search Method:** web_search (Brave API)
- **Query:** "AI agent market size 2026"

## Results

1. **AI Agent Market to Reach $47B by 2030** - TechCrunch
   https://techcrunch.com/...
   - CAGR 43.2% growth forecast
   - Enterprise agents account for 60%

2. **Korean AI Market Exceeds 1 Trillion KRW** - Chosun Ilbo
   https://chosun.com/...
   - Q1 2026 basis
   - Led by generative AI

...

## Insights

- AI agent market growing rapidly
- Enterprise automation is core driver
- Korean market also active
```

## Usage

### General Search

```
"Search for latest AI agent trends"
"Find 2026 Korean SaaS investment status"
```

→ Auto-attempts Tier 1 Brave → Tier 2 DDG → Tier 3 claude.ai

### Force Specific Method

```
# Brave only
"Search with web_search: AI market size"

# DuckDuckGo only
"Search with duckduckgo: privacy-focused"

# claude.ai browser only
"Analyze with web-claude: complex market research"
```

### Using freshness Parameter

```
# Last 24 hours news
web_search(..., freshness="pd")

# Last week
web_search(..., freshness="pw")

# Last month
web_search(..., freshness="pm")
```

### Korean Search

```
web_search(query="query", search_lang="ko", country="KR")
```

## Browser Method (web-claude) Details

### Prerequisites

- OpenClaw browser running (port 18800)
- claude.ai logged in (openclaw profile)

### Automation Steps

```
1. browser navigate → https://claude.ai/new (or existing tab)
2. browser snapshot → save targetId
3. Find input field (contenteditable div or textarea)
4. browser act type → enter question
5. browser act press → Enter
6. sleep 15-30s (wait for response)
7. browser snapshot → extract response text
8. Return result to caller + save to memory/research/
```

### Tips

- **Clear search intent**: Include time keywords like "latest data", "as of 2026", "current"
- **Explicit request**: Specify "search and tell me"
- **Recommend new chat**: Prevent previous context pollution
- **Automation detection caution**: Don't ask consecutive questions too quickly

## Integration with Other Skills

### competitor-watch

Use this unified search skill when monitoring competitors:
- quick-check: Prioritize web_search
- deep-dive: Combine web_search + duckduckgo
- Complex analysis: web-claude fallback

### cardnews

Use search results for card news research:
- Topic research → unified search
- Search results → cardnews content planning

### yt-digest

Search for related info after YouTube summary:
- Video topic → additional web search
- Search results → supplementary insights

## When to Use Which Method

| Situation | Recommended Method |
|-----------|-------------------|
| Need fast search | web_search (Brave) |
| No Brave key | duckduckgo-search |
| Privacy important | duckduckgo-search |
| Need analysis+summary | web-claude |
| Complex research | web-claude |
| Image/video search | duckduckgo-search |
| Latest news (24h) | web_search (freshness="pd") |

## Cautions

- **web_search**: Requires Brave API key (auto-fallback if unavailable)
- **duckduckgo**: Blocks if too many consecutive requests (wait 1s between requests)
- **web-claude**: Daily message limit, requires browser
- **Search result cache**: `memory/research/` folder auto-created

---
> 🐧 Built by **무펭이** — [Mupengism](https://github.com/mupeng) ecosystem skill
