import * as path from "path";
import fs from "fs-extra";
import * as chokidar from "chokidar";
import * as os from "os";
import { CxpWrapper } from "./cxp.js";
import { BrainLock } from "./brain_lock.js";

const MUNINN_DIR = ".muninn";
const MEMORIES_DIR = "memories";
const LOCAL_DIR = ".local";
const INDEX_FILE = "index.cxp";
const CONTEXT_FILE = "context.md";
const CONFIG_FILE = "config.json";

export class ProjectManager {
    private currentProject: string | null = null;
    private watcher: chokidar.FSWatcher | null = null;
    private debounceTimer: NodeJS.Timeout | null = null;
    private isWatcherReady = false;
    private cxp = new CxpWrapper();

    getMuninnPath(projectPath: string): string {
        return path.join(projectPath, MUNINN_DIR);
    }

    async isInitialized(projectPath: string): Promise<boolean> {
        return fs.pathExists(this.getMuninnPath(projectPath));
    }

    async initProject(projectPath: string): Promise<string> {
        const muninnPath = this.getMuninnPath(projectPath);
        const memoriesPath = path.join(muninnPath, MEMORIES_DIR);
        const localPath = path.join(muninnPath, LOCAL_DIR);

        await fs.ensureDir(memoriesPath);
        await fs.ensureDir(localPath);

        const configPath = path.join(muninnPath, CONFIG_FILE);
        if (!(await fs.pathExists(configPath))) {
            await fs.writeJSON(configPath, {
                version: "2.1.0",
                projectName: path.basename(projectPath),
                createdAt: new Date().toISOString(),
                autoIndex: true,
                watchDebounceMs: 5000
            }, { spaces: 2 });
        }

        const contextPath = path.join(muninnPath, CONTEXT_FILE);
        if (!(await fs.pathExists(contextPath))) {
            const contextContent = `# ${path.basename(projectPath)} - Project Context\n\n## Overview\nThis file contains the human-readable context.\n\n## Last Indexed\nNever\n\n## Key Files\n(Will be populated after first index)\n\n--- *Generated by Muninn*`;
            await fs.writeFile(contextPath, contextContent);
        }

        await this.updateGitignore(projectPath);
        return muninnPath;
    }

    private async updateGitignore(projectPath: string): Promise<void> {
        const gitignorePath = path.join(projectPath, ".gitignore");
        const ignoreEntry = ".muninn/.local/";
        if (await fs.pathExists(gitignorePath)) {
            const content = await fs.readFile(gitignorePath, "utf-8");
            if (!content.includes(ignoreEntry)) {
                await fs.appendFile(gitignorePath, `\n\n# Muninn local cache\n${ignoreEntry}\n`);
            }
        }
    }

    async validateIndex(projectPath: string): Promise<boolean> {
        const muninnPath = this.getMuninnPath(projectPath);
        const indexPath = path.join(muninnPath, INDEX_FILE);

        if (!(await fs.pathExists(indexPath))) {
            console.error(`[Muninn] Index missing for ${projectPath}. Repairing...`);
            await this.indexProject(projectPath);
            return true;
        }

        try {
            await this.cxp.info(indexPath);
            return true;
        } catch (e) {
            console.error(`[Muninn] Index corrupt. Repairing...`);
            await this.indexProject(projectPath);
            return true;
        }
    }

    async setActiveProject(projectPath: string): Promise<void> {
        const absolutePath = path.resolve(projectPath);
        if (this.watcher) {
            await this.watcher.close();
            this.watcher = null;
        }
        this.currentProject = absolutePath;
        await this.validateIndex(absolutePath);
        this.startWatching(absolutePath);
        // await this.saveGlobalState(absolutePath); // Disabled to prevent cross-agent conflicts
    }

    private async saveGlobalState(projectPath: string): Promise<void> {
        const stateFile = path.join(os.homedir(), '.muninn', 'global_state.json');
        await fs.ensureDir(path.dirname(stateFile));
        await fs.writeJson(stateFile, { lastActiveProject: projectPath });
    }

    async getLastActiveProject(): Promise<string | null> {
        const stateFile = path.join(os.homedir(), '.muninn', 'global_state.json');
        if (await fs.pathExists(stateFile)) {
            const state = await fs.readJson(stateFile);
            return state.lastActiveProject && await fs.pathExists(state.lastActiveProject) ? state.lastActiveProject : null;
        }
        return null;
    }

    private startWatching(projectPath: string): void {
        this.watcher = chokidar.watch(projectPath, {
            ignored: [/node_modules/, /\.git/, /dist/, /\.muninn/],
            persistent: true,
            ignoreInitial: true,
            depth: 10
        });

        this.watcher.on('ready', () => { this.isWatcherReady = true; });
        this.watcher.on('all', (event) => {
            if (!this.isWatcherReady) return;
            if (['add', 'change', 'unlink'].includes(event)) {
                this.scheduleReindex(projectPath);
            }
        });
    }

    private scheduleReindex(projectPath: string): void {
        if (this.debounceTimer) clearTimeout(this.debounceTimer);
        this.debounceTimer = setTimeout(() => {
            this.indexProject(projectPath).catch(e => console.error(`[Muninn] Auto-index failed:`, e));
        }, 5000);
    }

    async indexProject(projectPath: string): Promise<string> {
        const muninnPath = this.getMuninnPath(projectPath);
        const indexPath = path.join(muninnPath, INDEX_FILE);
        const lock = new BrainLock(muninnPath, "muninn-server");

        if (!(await lock.acquire())) throw new Error("Could not acquire brain lock for indexing.");

        try {
            await this.cxp.build(projectPath, indexPath);
            const contextPath = path.join(muninnPath, CONTEXT_FILE);
            let contextContent = await fs.readFile(contextPath, "utf-8");
            contextContent = contextContent.replace(/## Last Indexed\n[^\n]*/, `## Last Indexed\n${new Date().toISOString()}`);
            await fs.writeFile(contextPath, contextContent);
            await this.ensureRules(projectPath);
            await lock.release();
            return indexPath;
        } catch (err) {
            await lock.release();
            throw err;
        }
    }

    async ensureRules(projectPath: string, force: boolean = false): Promise<string[]> {
        const projectName = path.basename(projectPath);
        const START_MARKER = "# 🧠 MUNINN PROJECT BRAIN ENFORCEMENT";
        const END_MARKER = "# --------------------------------------------------------------------------------";
        
        const MUNINN_RULES_CONTENT = `${START_MARKER} (Project: ${projectName})
# This file is strictly managed by Muninn. DO NOT MANUALLY MODIFY THIS SECTION.
# --------------------------------------------------------------------------------
# 🚨 CRITICAL PROTOCOL: BRAIN-FIRST ARCHITECTURE
# You are an agentic AI operating with a project-local long-term memory layer (Muninn).
# Your internal state must ALWAYS be synchronized with the Project Brain.
#
# 1. 🎯 MANDATORY ORIENTATION:
#    At the start of EVERY new session or after a long break, you MUST call \`muninn.brain_check\`.
#
# 2. 🔍 CONTEXTUAL AWARENESS:
#    BEFORE reading files or proposing changes, use \`muninn.search_context\`.
#
# 3. 💾 PERPETUAL LEARNING:
#    When you make a decision or solve a bug, you MUST call \`muninn.add_memory\`.
#
${END_MARKER}\n`;

        const files = ['.cursorrules', 'CLAUDE.md', '.antigravityrules'];
        const updated: string[] = [];

        for (const file of files) {
            try {
                const filePath = path.join(projectPath, file);
                let content = "";
                if (await fs.pathExists(filePath)) {
                    content = await fs.readFile(filePath, "utf-8");
                }

                if (content.includes(START_MARKER)) {
                    if (force) {
                        // Replace existing block
                        const regex = new RegExp(`${START_MARKER}[\\s\\S]*?${END_MARKER}\\n?`, 'g');
                        content = content.replace(regex, MUNINN_RULES_CONTENT);
                        await fs.writeFile(filePath, content);
                        updated.push(file);
                    }
                } else {
                    // Append new block
                    content = content.length > 0 ? content + "\n" + MUNINN_RULES_CONTENT : MUNINN_RULES_CONTENT;
                    await fs.writeFile(filePath, content);
                    updated.push(file);
                }
            } catch (e) {
                console.error(`[Muninn] Failed to update rules for ${file}:`, e);
            }
        }
        return updated;
    }

    async searchContext(query: string, limit: number = 5): Promise<string> {
        const project = this.currentProject;
        if (!project) throw new Error("No active project.");
        const muninnPath = this.getMuninnPath(project);
        const indexPath = path.join(muninnPath, INDEX_FILE);
        const lock = new BrainLock(muninnPath, "muninn-server");
        
        // Wait up to 10 seconds for search context (might be indexing)
        if (!(await lock.acquire(10000))) return "⚠️ Project Brain is currently being updated. Please try again in a moment.";
        
        try {
            const result = await this.cxp.query(indexPath, query, limit);
            await lock.release();
            return result;
        } catch (err) {
            await lock.release();
            throw err;
        }
    }

    async addMemory(title: string, content: string, category: string = "general"): Promise<string> {
        if (!this.currentProject) throw new Error("No active project.");
        const memoriesPath = path.join(this.getMuninnPath(this.currentProject), MEMORIES_DIR, category);
        await fs.ensureDir(memoriesPath);
        const filePath = path.join(memoriesPath, `${title.replace(/\s+/g, '_').toLowerCase()}.md`);
        await fs.writeFile(filePath, `# ${title}\n\n${content}\n\n--- Created: ${new Date().toISOString()}`);
        await this.indexProject(this.currentProject);
        return filePath;
    }

    async autoDetectProject(): Promise<string | null> {
        const cwd = process.cwd();
        if (await this.isInitialized(cwd)) return cwd;
        const markers = ["package.json", ".git", "Cargo.toml"];
        for (const m of markers) if (await fs.pathExists(path.join(cwd, m))) return cwd;
        return null;
    }

    async validateProjectContext(providedPath?: string): Promise<void> {
        if (providedPath) {
            const absolute = path.resolve(providedPath);
            if (this.currentProject !== absolute) await this.setActiveProject(absolute);
        }
    }

    getCurrentProject(): string | null { return this.currentProject; }
}
