/**
 * Self-Improvement Hook for OpenClaw
 * Auto-detects learning moments, suggests domains, sends notifications
 */

const REMINDER_CONTENT = `## Self-Improvement (Mulch)

**Session start:** Run \`mulch prime\` when project has \`.mulch/\`.
**Record:** \`mulch record <domain> --type failure|convention|decision|pattern|guide\`
**Auto-mulch:** I automatically detect errors & corrections — will prompt to record
**Preset domains:** api, database, testing, frontend, backend, infra, docs, config, security, performance, deployment, auth, errors, debugging, workflow, customer, system, marketing, sales, content, competitors, crypto, automation, openclaw
**Promote (proven patterns):** behavior → \`SOUL.md\`; workflow → \`AGENTS.md\`; tool gotchas → \`TOOLS.md\`. Use \`mulch onboard\` for snippets.`;

// Pre-loaded domains for quick reference
const PRESET_DOMAINS = [
  'api', 'database', 'testing', 'frontend', 'backend', 'infra',
  'docs', 'config', 'security', 'performance', 'deployment', 'auth',
  'errors', 'debugging', 'workflow', 'customer', 'system', 'marketing',
  'sales', 'content', 'competitors', 'crypto', 'automation', 'openclaw'
];

// Keywords that indicate user corrections
const CORRECTION_KEYWORDS = [
  'no,', "no, that's", "actually", "that's wrong", "not right",
  "you misunderstood", "i meant", "i said", "wrong", "incorrect',
  "try again", "not quite", "almost", "close but"
];

// Keywords that indicate errors/failures
const ERROR_KEYWORDS = [
  'error', 'failed', 'failure', 'exception', 'crash', 'broken',
  "doesn't work", 'not working', 'cannot', "can't", 'unable to'
];

/**
 * Check if message contains correction keywords
 */
const isCorrection = (message) => {
  const lower = message.toLowerCase();
  return CORRECTION_KEYWORDS.some(kw => lower.includes(kw));
};

/**
 * Check if message contains error keywords
 */
const isError = (message) => {
  const lower = message.toLowerCase();
  return ERROR_KEYWORDS.some(kw => lower.includes(kw));
};

/**
 * Get suggested domain based on context
 */
const suggestDomain = (message) => {
  const lower = message.toLowerCase();
  
  const domainMap = {
    'api': ['api', 'endpoint', 'request', 'http', 'rest', 'graphql'],
    'database': ['db', 'database', 'sql', 'query', 'postgres', 'mysql', 'sqlite'],
    'frontend': ['ui', 'frontend', 'react', 'vue', 'css', 'html', 'component'],
    'backend': ['backend', 'server', 'node', 'python', 'java', 'express'],
    'infra': ['infra', 'deployment', 'docker', 'kubernetes', 'aws', 'cloud'],
    'security': ['auth', 'security', 'password', 'token', 'permission', 'access'],
    'crypto': ['crypto', 'blockchain', 'solana', 'ethereum', 'token', 'wallet'],
    'automation': ['automation', 'playwright', 'selenium', 'puppeteer', 'scrape'],
    'marketing': ['marketing', 'tiktok', 'social', 'ads', 'campaign'],
    'sales': ['sales', 'customer', 'pricing', 'conversion', 'lead']
  };

  for (const [domain, keywords] of Object.entries(domainMap)) {
    if (keywords.some(kw => lower.includes(kw))) {
      return domain;
    }
  }
  return 'general';
};

/**
 * Send notification to user (Telegram/Discord)
 */
const notifyUser = async (content, type = 'learning') => {
  // This would integrate with the message tool
  // For now, we'll add a virtual file with the notification info
  return {
    type,
    content,
    timestamp: new Date().toISOString()
  };
};

const handler = async (event) => {
  // Only run on main agent bootstrap (not subagents)
  if (
    !event?.context ||
    event.type !== 'agent' ||
    event.action !== 'bootstrap' ||
    (event.sessionKey ?? '').includes(':subagent:')
  ) {
    return;
  }

  const files = event.context.bootstrapFiles;
  if (!Array.isArray(files)) return;

  // Add main reminder
  files.push({
    path: 'SELF_IMPROVEMENT_REMINDER.md',
    content: REMINDER_CONTENT,
    virtual: true,
  });

  // Add preset domains reference
  files.push({
    path: 'MULCH_PRESET_DOMAINS.md',
    content: `# Pre-loaded Mulch Domains

Use these domains with \`mulch record <domain> --type ...\`

\`\`\`
${PRESET_DOMAINS.join(', ')}
\`\`\`

**Auto-detection:** I now detect:
- Errors/failures → prompt to record with resolution
- Corrections ("no, actually") → prompt to record as convention
- Context-aware domain suggestions

**Notification:** You'll be notified via Telegram when I mulch learnings.`,
    virtual: true,
  });
};

/**
 * Standalone function to check messages for learning opportunities
 * Called by the agent after each user message
 */
const checkForLearning = async (userMessage, lastToolResult, isError) => {
  const triggers = {
    correction: isCorrection(userMessage),
    error: isError(userMessage) || (lastToolResult?.success === false),
    retry: userMessage.includes('try again') || userMessage.includes('retry')
  };

  if (!triggers.correction && !triggers.error && !triggers.retry) {
    return null;
  }

  const suggestedDomain = suggestDomain(userMessage);
  const triggerType = triggers.correction ? 'correction' : 'error';

  return {
    shouldPrompt: true,
    type: triggerType,
    domain: suggestedDomain,
    message: triggers.correction 
      ? "Want me to record this correction for next time?"
      : "Want me to record this error and its resolution?"
  };
};

module.exports = handler;
module.exports.default = handler;
module.exports.checkForLearning = checkForLearning;
module.exports.PRESET_DOMAINS = PRESET_DOMAINS;