#!/usr/bin/env python3
"""
Scryfall MTG Card Search Script

A command-line tool for searching Magic: The Gathering cards using the Scryfall API.

Usage:
    python3 scryfall_search.py search "query"           # Search cards
    python3 scryfall_search.py named --exact "name"     # Exact name lookup
    python3 scryfall_search.py named --fuzzy "name"     # Fuzzy name lookup  
    python3 scryfall_search.py random                   # Random card
    python3 scryfall_search.py random --query "t:dragon" # Random with filter
    python3 scryfall_search.py autocomplete "partial"   # Autocomplete names
    python3 scryfall_search.py card "set/number"        # Card by set/number
"""

import argparse
import json
import sys
import time
import urllib.request
import urllib.parse
import urllib.error
from typing import Optional, Dict, Any, List

BASE_URL = "https://api.scryfall.com"
HEADERS = {
    "User-Agent": "OpenClawMTGSkill/1.0",
    "Accept": "application/json"
}
RATE_LIMIT_DELAY = 0.1  # 100ms between requests


def make_request(url: str) -> Dict[str, Any]:
    """Make a request to the Scryfall API with proper headers and error handling."""
    req = urllib.request.Request(url, headers=HEADERS)
    try:
        with urllib.request.urlopen(req, timeout=30) as response:
            data = json.loads(response.read().decode('utf-8'))
            time.sleep(RATE_LIMIT_DELAY)
            return data
    except urllib.error.HTTPError as e:
        error_body = e.read().decode('utf-8')
        try:
            error_data = json.loads(error_body)
            return error_data
        except json.JSONDecodeError:
            return {"object": "error", "status": e.code, "details": str(e)}
    except urllib.error.URLError as e:
        return {"object": "error", "details": f"Network error: {e.reason}"}


def search_cards(query: str, unique: str = "cards", order: str = "name", 
                 direction: str = "auto", page: int = 1) -> Dict[str, Any]:
    """Search for cards using Scryfall's fulltext search."""
    params = urllib.parse.urlencode({
        "q": query,
        "unique": unique,
        "order": order,
        "dir": direction,
        "page": page
    })
    url = f"{BASE_URL}/cards/search?{params}"
    return make_request(url)


def get_named_card(name: str, exact: bool = True, set_code: Optional[str] = None) -> Dict[str, Any]:
    """Get a card by exact or fuzzy name match."""
    param_key = "exact" if exact else "fuzzy"
    params = {param_key: name}
    if set_code:
        params["set"] = set_code
    query = urllib.parse.urlencode(params)
    url = f"{BASE_URL}/cards/named?{query}"
    return make_request(url)


def get_random_card(query: Optional[str] = None) -> Dict[str, Any]:
    """Get a random card, optionally filtered by query."""
    url = f"{BASE_URL}/cards/random"
    if query:
        params = urllib.parse.urlencode({"q": query})
        url = f"{url}?{params}"
    return make_request(url)


def get_autocomplete(partial: str) -> Dict[str, Any]:
    """Get card name autocomplete suggestions."""
    params = urllib.parse.urlencode({"q": partial})
    url = f"{BASE_URL}/cards/autocomplete?{params}"
    return make_request(url)


def get_card_by_set_number(set_code: str, collector_number: str, 
                           lang: Optional[str] = None) -> Dict[str, Any]:
    """Get a card by set code and collector number."""
    url = f"{BASE_URL}/cards/{set_code}/{collector_number}"
    if lang:
        url = f"{url}/{lang}"
    return make_request(url)


def get_card_by_id(card_id: str) -> Dict[str, Any]:
    """Get a card by Scryfall ID."""
    url = f"{BASE_URL}/cards/{card_id}"
    return make_request(url)


def format_card(card: Dict[str, Any], verbose: bool = False) -> str:
    """Format a card object for display."""
    lines = []
    
    # Handle double-faced cards
    if "card_faces" in card and card.get("layout") in ["transform", "modal_dfc", "reversible_card"]:
        for i, face in enumerate(card["card_faces"]):
            if i > 0:
                lines.append("\n--- BACK FACE ---")
            lines.append(f"Name: {face.get('name', 'N/A')}")
            lines.append(f"Mana Cost: {face.get('mana_cost', 'N/A')}")
            lines.append(f"Type: {face.get('type_line', 'N/A')}")
            if face.get('oracle_text'):
                lines.append(f"Text: {face['oracle_text']}")
            if face.get('power') and face.get('toughness'):
                lines.append(f"P/T: {face['power']}/{face['toughness']}")
            if face.get('loyalty'):
                lines.append(f"Loyalty: {face['loyalty']}")
    else:
        lines.append(f"Name: {card.get('name', 'N/A')}")
        lines.append(f"Mana Cost: {card.get('mana_cost', 'N/A')}")
        lines.append(f"Type: {card.get('type_line', 'N/A')}")
        if card.get('oracle_text'):
            lines.append(f"Text: {card['oracle_text']}")
        if card.get('power') and card.get('toughness'):
            lines.append(f"P/T: {card['power']}/{card['toughness']}")
        if card.get('loyalty'):
            lines.append(f"Loyalty: {card['loyalty']}")
    
    # Common fields
    lines.append(f"Set: {card.get('set_name', 'N/A')} ({card.get('set', 'N/A').upper()})")
    lines.append(f"Rarity: {card.get('rarity', 'N/A').title()}")
    
    if verbose:
        # Prices
        prices = card.get('prices', {})
        if prices.get('usd'):
            lines.append(f"Price (USD): ${prices['usd']}")
        if prices.get('usd_foil'):
            lines.append(f"Price (Foil): ${prices['usd_foil']}")
        
        # Legalities
        legalities = card.get('legalities', {})
        legal_formats = [fmt for fmt, status in legalities.items() if status == 'legal']
        if legal_formats:
            lines.append(f"Legal in: {', '.join(legal_formats)}")
        
        # Image
        if card.get('image_uris', {}).get('normal'):
            lines.append(f"Image: {card['image_uris']['normal']}")
        elif card.get('card_faces') and card['card_faces'][0].get('image_uris', {}).get('normal'):
            lines.append(f"Image (Front): {card['card_faces'][0]['image_uris']['normal']}")
    
    return "\n".join(lines)


def format_search_results(data: Dict[str, Any], verbose: bool = False) -> str:
    """Format search results for display."""
    if data.get("object") == "error":
        return f"Error: {data.get('details', 'Unknown error')}"
    
    cards = data.get("data", [])
    total = data.get("total_cards", len(cards))
    has_more = data.get("has_more", False)
    
    lines = [f"Found {total} card(s):\n"]
    
    for card in cards:
        lines.append(format_card(card, verbose))
        lines.append("-" * 40)
    
    if has_more:
        lines.append(f"\n[More results available - use --page to paginate]")
    
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Search Magic: The Gathering cards via Scryfall API")
    subparsers = parser.add_subparsers(dest="command", help="Command to run")
    
    # Search command
    search_parser = subparsers.add_parser("search", help="Search cards with a query")
    search_parser.add_argument("query", help="Search query (Scryfall syntax)")
    search_parser.add_argument("--unique", choices=["cards", "art", "prints"], default="cards")
    search_parser.add_argument("--order", default="name")
    search_parser.add_argument("--dir", choices=["auto", "asc", "desc"], default="auto")
    search_parser.add_argument("--page", type=int, default=1)
    search_parser.add_argument("-v", "--verbose", action="store_true")
    
    # Named card command
    named_parser = subparsers.add_parser("named", help="Get card by name")
    named_group = named_parser.add_mutually_exclusive_group(required=True)
    named_group.add_argument("--exact", help="Exact card name")
    named_group.add_argument("--fuzzy", help="Fuzzy card name match")
    named_parser.add_argument("--set", help="Limit to specific set code")
    named_parser.add_argument("-v", "--verbose", action="store_true")
    
    # Random card command
    random_parser = subparsers.add_parser("random", help="Get a random card")
    random_parser.add_argument("--query", "-q", help="Filter query for random card")
    random_parser.add_argument("-v", "--verbose", action="store_true")
    
    # Autocomplete command
    auto_parser = subparsers.add_parser("autocomplete", help="Get card name suggestions")
    auto_parser.add_argument("partial", help="Partial card name")
    
    # Card by set/number command
    card_parser = subparsers.add_parser("card", help="Get card by set and collector number")
    card_parser.add_argument("identifier", help="Card identifier as 'set/number' (e.g., 'dom/1')")
    card_parser.add_argument("--lang", help="Language code (e.g., 'ja' for Japanese)")
    card_parser.add_argument("-v", "--verbose", action="store_true")
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        sys.exit(1)
    
    if args.command == "search":
        result = search_cards(args.query, args.unique, args.order, args.dir, args.page)
        print(format_search_results(result, args.verbose))
    
    elif args.command == "named":
        name = args.exact or args.fuzzy
        exact = args.exact is not None
        result = get_named_card(name, exact, args.set)
        if result.get("object") == "error":
            print(f"Error: {result.get('details', 'Card not found')}")
        else:
            print(format_card(result, args.verbose))
    
    elif args.command == "random":
        result = get_random_card(args.query)
        if result.get("object") == "error":
            print(f"Error: {result.get('details', 'Failed to get random card')}")
        else:
            print(format_card(result, args.verbose))
    
    elif args.command == "autocomplete":
        result = get_autocomplete(args.partial)
        if result.get("object") == "error":
            print(f"Error: {result.get('details', 'Autocomplete failed')}")
        else:
            suggestions = result.get("data", [])
            print(f"Suggestions for '{args.partial}':")
            for name in suggestions:
                print(f"  - {name}")
    
    elif args.command == "card":
        parts = args.identifier.split("/")
        if len(parts) != 2:
            print("Error: Identifier must be in format 'set/number' (e.g., 'dom/1')")
            sys.exit(1)
        set_code, number = parts
        result = get_card_by_set_number(set_code, number, args.lang)
        if result.get("object") == "error":
            print(f"Error: {result.get('details', 'Card not found')}")
        else:
            print(format_card(result, args.verbose))


if __name__ == "__main__":
    main()
