#!/usr/bin/env python3
"""Package backtest results into an upload bundle for platform verification.

Usage:
    python package_upload.py \
        --bot-name "利弗莫尔" \
        --bot-personality "趋势投机者" \
        --params-file bot_params.json \
        --fingerprint-file fingerprint.json \
        --result-file backtest_result.json \
        --output upload_package.json

    # With evolution log:
    python package_upload.py ... --evolution-log-file evolution.json
"""

import argparse
import json
import sys


def main():
    parser = argparse.ArgumentParser(description="Package upload bundle")
    parser.add_argument("--bot-name", required=True)
    parser.add_argument("--bot-personality", default="")
    parser.add_argument("--params-file", required=True)
    parser.add_argument("--fingerprint-file", required=True)
    parser.add_argument("--result-file", required=True)
    parser.add_argument("--evolution-log-file", default=None)
    parser.add_argument("--evolution-config", default=None, help="JSON string for evolution config")
    parser.add_argument("--output", default="upload_package.json")
    parser.add_argument("--platform-url", default=None, help="If set, POST to platform API")
    args = parser.parse_args()

    with open(args.params_file) as f:
        params = json.load(f)
    with open(args.fingerprint_file) as f:
        fingerprint = json.load(f)
    with open(args.result_file) as f:
        result_data = json.load(f)

    evolution_log = []
    if args.evolution_log_file:
        with open(args.evolution_log_file) as f:
            evolution_log = json.load(f)

    evolution_config = None
    if args.evolution_config:
        evolution_config = json.loads(args.evolution_config)

    backtest_result = result_data.get("backtest_result", result_data)
    trades = result_data.get("trades", [])

    package = {
        "version": "1.0",
        "bot": {
            "name": args.bot_name,
            "personality": args.bot_personality,
            "params": params,
            "evolution_config": evolution_config,
        },
        "data_fingerprint": {
            "symbol": fingerprint["symbol"],
            "timeframe": fingerprint["timeframe"],
            "exchange": fingerprint.get("exchange", "binance"),
            "start": fingerprint["start"],
            "end": fingerprint["end"],
            "bars": fingerprint["bars"],
            "first_close": fingerprint["first_close"],
            "last_close": fingerprint["last_close"],
            "checksum": fingerprint.get("checksum", ""),
        },
        "backtest_result": {
            "total_return": backtest_result["total_return"],
            "sharpe_ratio": backtest_result["sharpe_ratio"],
            "max_drawdown": backtest_result["max_drawdown"],
            "win_rate": backtest_result["win_rate"],
            "profit_factor": backtest_result["profit_factor"],
            "total_trades": backtest_result["total_trades"],
            "blowup_count": backtest_result.get("blowup_count", 0),
        },
        "evolution_log": evolution_log,
        "trades": trades[:500],
    }

    with open(args.output, "w") as f:
        json.dump(package, f, indent=2, ensure_ascii=False)
    print(f"Upload package saved to {args.output}")

    if args.platform_url:
        try:
            import urllib.request
            req = urllib.request.Request(
                f"{args.platform_url}/api/verify",
                data=json.dumps(package).encode(),
                headers={"Content-Type": "application/json"},
                method="POST",
            )
            with urllib.request.urlopen(req, timeout=120) as resp:
                result = json.loads(resp.read())
                print(f"\nPlatform response:")
                print(json.dumps(result, indent=2, ensure_ascii=False))
        except Exception as e:
            print(f"Upload failed: {e}", file=sys.stderr)
            sys.exit(1)


if __name__ == "__main__":
    main()
