#!/usr/bin/env npx tsx
/**
 * Test swap script: Split USDC into WELL and MORPHO tokens for testing compound
 * Usage: npx tsx test-swap.ts [amount]
 * 
 * This swaps half the USDC to WELL and half to MORPHO, simulating
 * claimed rewards so you can test the compound flow.
 */

import {
  loadConfig,
  getClients,
  USDC_ADDRESS,
  ERC20_ABI,
  formatUSDC,
  parseUSDC,
  verifyContracts,
  waitForTransaction,
  logTransaction,
  handleError,
  rateLimitedFetch,
  approveAndVerify,
  getFreshNonce,
} from './config.js';
import { type Address, type Hex, formatUnits } from 'viem';

// Common token addresses on Base
const WELL_ADDRESS = '0xA88594D404727625A9437C3f886C7643872296AE' as Address;
const MORPHO_ADDRESS = '0xBAa5CC21fd487B8Fcc2F632f3F4E8D37262a0842' as Address;

// Odos Router V2 on Base
const ODOS_ROUTER = '0x19cEeAd7105607Cd444F5ad10dd51356436095a1' as Address;
const BASE_CHAIN_ID = 8453;

interface OdosQuoteResponse {
  pathId: string;
  outAmounts: string[];
  gasEstimate: number;
  outValues: number[];
}

interface OdosAssembleResponse {
  transaction: {
    to: string;
    data: string;
    value: string;
    gas: number;
  };
  outputTokens: Array<{ amount: string }>;
}

async function getOdosQuote(
  tokenIn: Address,
  amountIn: bigint,
  tokenOut: Address,
  userAddress: Address
): Promise<OdosQuoteResponse | null> {
  const response = await rateLimitedFetch('https://api.odos.xyz/sor/quote/v2', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      chainId: BASE_CHAIN_ID,
      inputTokens: [{ tokenAddress: tokenIn, amount: amountIn.toString() }],
      outputTokens: [{ tokenAddress: tokenOut, proportion: 1 }],
      slippageLimitPercent: 1,
      userAddr: userAddress,
    }),
  });
  
  if (!response.ok) {
    console.log(`  ⚠️ Odos quote failed: ${response.status}`);
    return null;
  }
  
  const data = await response.json() as OdosQuoteResponse;
  
  if (!data.pathId || !Array.isArray(data.outAmounts)) {
    console.log(`  ⚠️ Invalid Odos quote response`);
    return null;
  }
  
  return data;
}

async function assembleOdosTransaction(
  pathId: string,
  userAddress: Address
): Promise<OdosAssembleResponse | null> {
  const response = await rateLimitedFetch('https://api.odos.xyz/sor/assemble', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      userAddr: userAddress,
      pathId,
      simulate: false,
    }),
  });
  
  if (!response.ok) {
    console.log(`  ⚠️ Odos assemble failed: ${response.status}`);
    return null;
  }
  
  const data = await response.json() as OdosAssembleResponse;
  
  if (!data.transaction?.to || !data.transaction?.data) {
    console.log(`  ⚠️ Invalid Odos assemble response`);
    return null;
  }
  
  return data;
}

async function swapToken(
  publicClient: ReturnType<typeof getClients>['publicClient'],
  walletClient: ReturnType<typeof getClients>['walletClient'],
  account: ReturnType<typeof getClients>['account'],
  tokenIn: Address,
  tokenInSymbol: string,
  amountIn: bigint,
  tokenOut: Address,
  tokenOutSymbol: string
): Promise<boolean> {
  console.log(`\nSwapping ${formatUSDC(amountIn)} ${tokenInSymbol} → ${tokenOutSymbol}...`);
  
  // Get quote
  const quote = await getOdosQuote(tokenIn, amountIn, tokenOut, account.address);
  if (!quote) {
    console.log(`  ❌ Could not get quote`);
    return false;
  }
  
  const expectedOut = BigInt(quote.outAmounts[0]);
  const decimals = tokenOutSymbol === 'USDC' ? 6 : 18;
  console.log(`  Expected: ${formatUnits(expectedOut, decimals)} ${tokenOutSymbol}`);
  
  // Check and approve using the new verification flow
  const currentAllowance = await publicClient.readContract({
    address: tokenIn,
    abi: ERC20_ABI,
    functionName: 'allowance',
    args: [account.address, ODOS_ROUTER],
  });
  
  if (currentAllowance < amountIn) {
    console.log(`  Approving ${tokenInSymbol} for Odos router...`);
    try {
      const approveHash = await approveAndVerify(
        publicClient,
        walletClient,
        account,
        tokenIn,
        ODOS_ROUTER,
        amountIn,
        tokenInSymbol
      );
      console.log(`  ✅ Approved and verified! Tx: ${approveHash}`);
    } catch (err) {
      console.log(`  ❌ Approval failed: ${err instanceof Error ? err.message : String(err)}`);
      return false;
    }
  } else {
    console.log(`  ✅ Already approved`);
  }
  
  // Assemble transaction
  const assembled = await assembleOdosTransaction(quote.pathId, account.address);
  if (!assembled) {
    console.log(`  ❌ Could not assemble transaction`);
    return false;
  }
  
  // Execute swap
  console.log(`  Executing swap...`);
  const gasEstimate = BigInt(assembled.transaction.gas);
  const gasWithBuffer = gasEstimate + (gasEstimate * 50n / 100n);
  
  // Get fresh nonce to avoid stale cache issues
  const nonce = await getFreshNonce(publicClient, account.address);
  
  try {
    const swapHash = await walletClient.sendTransaction({
      to: assembled.transaction.to as Address,
      data: assembled.transaction.data as Hex,
      value: BigInt(assembled.transaction.value),
      gas: gasWithBuffer,
      nonce,
    });
    
    console.log(`  Tx: ${swapHash}`);
    const receipt = await waitForTransaction(publicClient, swapHash);
    
    if (receipt.status === 'success') {
      logTransaction('swap', swapHash, {
        tokenIn: tokenInSymbol,
        amountIn: formatUSDC(amountIn),
        tokenOut: tokenOutSymbol,
        expectedOut: formatUnits(expectedOut, decimals),
        test: true,
      });
      
      console.log(`  ✅ Swapped!`);
      return true;
    } else {
      console.log(`  ❌ Swap reverted`);
      return false;
    }
  } catch (err) {
    console.log(`  ❌ Swap failed: ${err instanceof Error ? err.message : String(err)}`);
    return false;
  }
}

async function main() {
  const args = process.argv.slice(2);
  let swapAmount: bigint;
  
  // Handle help flag
  if (args.includes('--help') || args.includes('-h')) {
    console.log('Usage: npx tsx test-swap.ts [amount]');
    console.log('\nSplits USDC into WELL and MORPHO tokens for testing compound.');
    console.log('\nExamples:');
    console.log('  npx tsx test-swap.ts       # Use half balance, up to $5');
    console.log('  npx tsx test-swap.ts 2     # Swap $2 total ($1 each)');
    process.exit(0);
  }
  
  if (args.length > 0 && !args[0].startsWith('-')) {
    swapAmount = parseUSDC(args[0]);
  } else {
    // Default: use half the USDC balance, up to $5
    swapAmount = 0n; // Will be set after checking balance
  }
  
  const config = loadConfig();
  const { publicClient, walletClient, account } = getClients(config);
  
  console.log('🧪 Test Swap: USDC → WELL + MORPHO\n');
  console.log(`Config loaded from: ~/.config/morpho-yield/config.json`);
  console.log(`Wallet: ${account.address}\n`);
  
  // Verify contracts
  console.log('🔐 Verifying contracts...');
  try {
    await verifyContracts(publicClient);
    console.log('   ✅ Contracts verified\n');
  } catch (err) {
    handleError(err, 'Contract verification failed');
  }
  
  // Check USDC balance
  const usdcBalance = await publicClient.readContract({
    address: USDC_ADDRESS,
    abi: ERC20_ABI,
    functionName: 'balanceOf',
    args: [account.address],
  });
  
  console.log(`USDC balance: ${formatUSDC(usdcBalance)}`);
  
  if (usdcBalance < 100000n) { // Less than $0.10
    console.error('❌ Insufficient USDC for test swap (need at least $0.10)');
    process.exit(1);
  }
  
  // Determine swap amount
  if (swapAmount === 0n) {
    // Default: use half the balance, capped at $5
    const halfBalance = usdcBalance / 2n;
    const maxAmount = 5_000_000n; // $5
    swapAmount = halfBalance > maxAmount ? maxAmount : halfBalance;
  }
  
  if (swapAmount > usdcBalance) {
    console.error(`❌ Insufficient USDC. Have: ${formatUSDC(usdcBalance)}, want: ${formatUSDC(swapAmount)}`);
    process.exit(1);
  }
  
  // Check ETH for gas
  const ethBalance = await publicClient.getBalance({ address: account.address });
  if (ethBalance < BigInt(5e14)) { // 0.0005 ETH minimum
    console.error(`❌ Insufficient ETH for gas`);
    console.error(`   Available: ${(Number(ethBalance) / 1e18).toFixed(6)} ETH`);
    console.error(`   Recommended: at least 0.0005 ETH`);
    process.exit(1);
  }
  
  // Split into two halves
  const halfAmount = swapAmount / 2n;
  
  console.log(`\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
  console.log(`📋 Test Swap Plan`);
  console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
  console.log(`Swapping ${formatUSDC(halfAmount)} USDC → WELL`);
  console.log(`Swapping ${formatUSDC(halfAmount)} USDC → MORPHO`);
  console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
  
  // Swap USDC to WELL
  const wellSuccess = await swapToken(
    publicClient, walletClient, account,
    USDC_ADDRESS, 'USDC',
    halfAmount,
    WELL_ADDRESS, 'WELL'
  );
  
  // Swap USDC to MORPHO
  const morphoSuccess = await swapToken(
    publicClient, walletClient, account,
    USDC_ADDRESS, 'USDC',
    halfAmount,
    MORPHO_ADDRESS, 'MORPHO'
  );
  
  // Summary
  console.log(`\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
  console.log(`📊 Results`);
  console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
  console.log(`WELL swap: ${wellSuccess ? '✅' : '❌'}`);
  console.log(`MORPHO swap: ${morphoSuccess ? '✅' : '❌'}`);
  
  // Show final token balances
  const finalUSDC = await publicClient.readContract({
    address: USDC_ADDRESS, abi: ERC20_ABI, functionName: 'balanceOf', args: [account.address],
  });
  const finalWELL = await publicClient.readContract({
    address: WELL_ADDRESS, abi: ERC20_ABI, functionName: 'balanceOf', args: [account.address],
  });
  const finalMORPHO = await publicClient.readContract({
    address: MORPHO_ADDRESS, abi: ERC20_ABI, functionName: 'balanceOf', args: [account.address],
  });
  
  console.log(`\nFinal balances:`);
  console.log(`  USDC:   ${formatUSDC(finalUSDC)}`);
  console.log(`  WELL:   ${formatUnits(finalWELL, 18)}`);
  console.log(`  MORPHO: ${formatUnits(finalMORPHO, 18)}`);
  console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
  
  if (wellSuccess || morphoSuccess) {
    console.log(`\n✅ Done! Ready to test compound.`);
  } else {
    console.log(`\n❌ All swaps failed.`);
    process.exit(1);
  }
}

main().catch((err) => handleError(err, 'Test swap failed'));
