#!/usr/bin/env npx tsx
/**
 * Check and claim rewards from Morpho vault
 * Usage: 
 *   npx tsx rewards.ts           # Check claimable rewards
 *   npx tsx rewards.ts claim     # Claim all rewards
 */

import {
  loadConfig,
  getClients,
  VAULT_ADDRESS,
  formatUSDC,
} from './config.js';
import { type Address, type Hex, formatUnits } from 'viem';

// Merkl Distributor on Base
const MERKL_DISTRIBUTOR = '0x3Ef3D8bA38EBe18DB133cEc108f4D14CE00Dd9Ae' as Address;
const BASE_CHAIN_ID = 8453;

// Merkl Distributor ABI
const MERKL_ABI = [
  {
    inputs: [
      { name: 'users', type: 'address[]' },
      { name: 'tokens', type: 'address[]' },
      { name: 'amounts', type: 'uint256[]' },
      { name: 'proofs', type: 'bytes32[][]' },
    ],
    name: 'claim',
    outputs: [],
    stateMutability: 'nonpayable',
    type: 'function',
  },
] as const;

interface MerklReward {
  chain: { id: number };
  recipient: string;
  rewards: Array<{
    token: {
      address: string;
      symbol: string;
      decimals: number;
    };
    amount: string;
    claimed: string;
    pending: string;
    proofs: string[];
  }>;
}

interface TokenReward {
  token: Address;
  symbol: string;
  decimals: number;
  total: bigint;
  claimed: bigint;
  claimable: bigint;
  pending: bigint;
  proofs: Hex[];
}

async function fetchRewards(userAddress: string): Promise<TokenReward[]> {
  const url = `https://api.merkl.xyz/v4/users/${userAddress}/rewards?chainId=${BASE_CHAIN_ID}`;
  
  const response = await fetch(url);
  if (!response.ok) {
    throw new Error(`Merkl API error: ${response.status}`);
  }
  
  const data = await response.json() as MerklReward[];
  const rewards: TokenReward[] = [];
  
  for (const chainRewards of data) {
    if (chainRewards.chain.id !== BASE_CHAIN_ID) continue;
    
    for (const reward of chainRewards.rewards) {
      const total = BigInt(reward.amount);
      const claimed = BigInt(reward.claimed);
      const claimable = total - claimed;
      
      if (claimable > 0n || BigInt(reward.pending) > 0n) {
        rewards.push({
          token: reward.token.address as Address,
          symbol: reward.token.symbol,
          decimals: reward.token.decimals,
          total,
          claimed,
          claimable,
          pending: BigInt(reward.pending),
          proofs: reward.proofs as Hex[],
        });
      }
    }
  }
  
  return rewards;
}

async function checkRewards() {
  const config = loadConfig();
  const { account } = getClients(config);
  
  console.log('🌙 Moonwell Vault — Rewards Check\n');
  console.log(`Wallet: ${account.address}`);
  console.log(`Vault:  ${VAULT_ADDRESS}\n`);
  
  console.log('Fetching rewards from Merkl...\n');
  
  const rewards = await fetchRewards(account.address);
  
  if (rewards.length === 0) {
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log('No claimable rewards found.');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log('\n💡 Rewards accrue over time. Check back later!');
    return rewards;
  }
  
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('🎁 Available Rewards');
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  
  for (const reward of rewards) {
    const claimableFormatted = formatUnits(reward.claimable, reward.decimals);
    const pendingFormatted = formatUnits(reward.pending, reward.decimals);
    
    console.log(`\n${reward.symbol}:`);
    console.log(`  Claimable: ${parseFloat(claimableFormatted).toFixed(6)} ${reward.symbol}`);
    if (reward.pending > 0n) {
      console.log(`  Pending:   ${parseFloat(pendingFormatted).toFixed(6)} ${reward.symbol} (next update)`);
    }
  }
  
  console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('\n💡 Run `npx tsx rewards.ts claim` to claim rewards');
  
  return rewards;
}

async function claimRewards() {
  const config = loadConfig();
  const { publicClient, walletClient, account } = getClients(config);
  
  console.log('🌙 Moonwell Vault — Claim Rewards\n');
  console.log(`Wallet: ${account.address}\n`);
  
  console.log('Fetching claimable rewards...\n');
  
  const rewards = await fetchRewards(account.address);
  const claimable = rewards.filter(r => r.claimable > 0n);
  
  if (claimable.length === 0) {
    console.log('❌ No rewards available to claim.');
    return;
  }
  
  // Prepare claim data
  const users: Address[] = [];
  const tokens: Address[] = [];
  const amounts: bigint[] = [];
  const proofs: Hex[][] = [];
  
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('📋 Claiming:');
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  
  for (const reward of claimable) {
    users.push(account.address);
    tokens.push(reward.token);
    amounts.push(reward.total); // Use total amount for Merkl claim
    proofs.push(reward.proofs);
    
    const amountFormatted = formatUnits(reward.claimable, reward.decimals);
    console.log(`  ${parseFloat(amountFormatted).toFixed(6)} ${reward.symbol}`);
  }
  
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');
  
  // Check ETH for gas
  const ethBalance = await publicClient.getBalance({ address: account.address });
  if (ethBalance < BigInt(1e14)) {
    console.error(`❌ Insufficient ETH for gas`);
    console.error(`   Available: ${(Number(ethBalance) / 1e18).toFixed(6)} ETH`);
    process.exit(1);
  }
  
  console.log('📝 Submitting claim transaction...');
  
  const claimHash = await walletClient.writeContract({
    address: MERKL_DISTRIBUTOR,
    abi: MERKL_ABI,
    functionName: 'claim',
    args: [users, tokens, amounts, proofs],
  });
  
  console.log(`   Tx: ${claimHash}`);
  console.log('   Waiting for confirmation...');
  
  const receipt = await publicClient.waitForTransactionReceipt({ hash: claimHash });
  
  if (receipt.status === 'success') {
    console.log('   ✅ Rewards claimed!\n');
    
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log('🎉 Claim Complete!');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log(`View on BaseScan: https://basescan.org/tx/${claimHash}`);
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    
    console.log('\n💡 Rewards are now in your wallet.');
    console.log('   Run `npx tsx compound.ts` to convert and re-deposit as USDC.');
  } else {
    console.error('   ❌ Transaction failed');
    process.exit(1);
  }
}

async function main() {
  const args = process.argv.slice(2);
  const action = args[0]?.toLowerCase();
  
  if (action === 'claim') {
    await claimRewards();
  } else {
    await checkRewards();
  }
}

main().catch((err) => {
  console.error('❌ Error:', err.message);
  process.exit(1);
});
