#!/usr/bin/env npx tsx
/**
 * Generate a beautiful status report for chat surfaces
 * Usage: npx tsx report.ts [--json|--plain]
 */

import {
  loadConfig,
  getClients,
  VAULT_ADDRESS,
  USDC_ADDRESS,
  VAULT_ABI,
  ERC20_ABI,
  formatUSDC,
  fetchVaultAPY,
  rateLimitedFetch,
  handleError,
} from './config.js';
import { type Address } from 'viem';

const WELL_ADDRESS = '0xA88594D404727625A9437C3f886C7643872296AE' as Address;
const MORPHO_ADDRESS = '0xBAa5CC21fd487B8Fcc2F632f3F4E8D37262a0842' as Address;

// Estimated APRs for reward tokens (from Merkl, can vary)
// Check https://api.merkl.xyz/v4/opportunities?chainId=8453 for current rates
const REWARD_APRS = {
  WELL: 0.003,   // ~0.3% APR in WELL rewards
  MORPHO: 0.003, // ~0.3% APR in MORPHO rewards
};

interface ReportData {
  wallet: string;
  positionUSDC: number;
  vaultAPY: number;
  totalAPY: number;
  wellBalance: number;
  morphoBalance: number;
  wellValueUSD: number;
  morphoValueUSD: number;
  totalRewardsUSD: number;
  ethBalance: number;
}

async function getTokenPrice(symbol: string): Promise<number> {
  const ids: Record<string, string> = {
    WELL: 'moonwell',
    MORPHO: 'morpho',
  };
  
  const id = ids[symbol];
  if (!id) return 0;
  
  try {
    const res = await rateLimitedFetch(
      `https://api.coingecko.com/api/v3/simple/price?ids=${id}&vs_currencies=usd`
    );
    
    if (!res.ok) return 0;
    
    const data = await res.json() as Record<string, { usd?: number }>;
    return data[id]?.usd ?? 0;
  } catch {
    return 0;
  }
}

async function gatherReportData(): Promise<ReportData> {
  const config = loadConfig();
  const { publicClient, account } = getClients(config);
  
  // Fetch all data in parallel
  const [
    shares,
    totalAssets,
    totalSupply,
    wellBalance,
    morphoBalance,
    ethBalance,
    wellPrice,
    morphoPrice,
    vaultAPY,
  ] = await Promise.all([
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'balanceOf',
      args: [account.address],
    }),
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'totalAssets',
    }),
    publicClient.readContract({
      address: VAULT_ADDRESS,
      abi: VAULT_ABI,
      functionName: 'totalSupply',
    }),
    publicClient.readContract({
      address: WELL_ADDRESS,
      abi: ERC20_ABI,
      functionName: 'balanceOf',
      args: [account.address],
    }),
    publicClient.readContract({
      address: MORPHO_ADDRESS,
      abi: ERC20_ABI,
      functionName: 'balanceOf',
      args: [account.address],
    }),
    publicClient.getBalance({ address: account.address }),
    getTokenPrice('WELL'),
    getTokenPrice('MORPHO'),
    fetchVaultAPY(),
  ]);
  
  // Calculate position value
  const positionValue = totalSupply > 0n
    ? (shares * totalAssets) / totalSupply
    : 0n;
  
  const positionUSDC = Number(positionValue) / 1e6;
  const wellBal = Number(wellBalance) / 1e18;
  const morphoBal = Number(morphoBalance) / 1e18;
  
  // Use fetched APY or fallback
  const baseAPY = vaultAPY > 0 ? vaultAPY : 4.09;
  
  // Total APY including rewards
  const totalAPY = baseAPY + (REWARD_APRS.WELL * 100) + (REWARD_APRS.MORPHO * 100);
  
  return {
    wallet: account.address,
    positionUSDC,
    vaultAPY: baseAPY,
    totalAPY,
    wellBalance: wellBal,
    morphoBalance: morphoBal,
    wellValueUSD: wellBal * wellPrice,
    morphoValueUSD: morphoBal * morphoPrice,
    totalRewardsUSD: (wellBal * wellPrice) + (morphoBal * morphoPrice),
    ethBalance: Number(ethBalance) / 1e18,
  };
}

function formatTelegramReport(data: ReportData): string {
  const shortWallet = `${data.wallet.slice(0, 6)}...${data.wallet.slice(-4)}`;
  
  let report = `🌜🌛 **Moonwell Yield Report**\n\n`;
  
  // Position
  report += `📊 **Position**\n`;
  report += `├ Value: **$${data.positionUSDC.toFixed(2)}**\n`;
  report += `├ Base APY: ${data.vaultAPY.toFixed(2)}%\n`;
  report += `└ Total APY: ~${data.totalAPY.toFixed(2)}%\n\n`;
  
  // Rewards
  if (data.totalRewardsUSD > 0.01) {
    report += `🎁 **Pending Rewards**\n`;
    if (data.wellBalance > 0) {
      report += `├ WELL: ${data.wellBalance.toFixed(2)} (~$${data.wellValueUSD.toFixed(2)})\n`;
    }
    if (data.morphoBalance > 0) {
      report += `├ MORPHO: ${data.morphoBalance.toFixed(4)} (~$${data.morphoValueUSD.toFixed(2)})\n`;
    }
    report += `└ Total: **$${data.totalRewardsUSD.toFixed(2)}**\n\n`;
  }
  
  // Estimated earnings
  const dailyEarnings = (data.positionUSDC * data.totalAPY / 100) / 365;
  const monthlyEarnings = dailyEarnings * 30;
  
  report += `💰 **Estimated Earnings**\n`;
  report += `├ Daily: ~$${dailyEarnings.toFixed(4)}\n`;
  report += `└ Monthly: ~$${monthlyEarnings.toFixed(2)}\n\n`;
  
  // Gas status
  const gasStatus = data.ethBalance > 0.001 ? '✅' : '⚠️';
  report += `⛽ Gas: ${gasStatus} ${data.ethBalance.toFixed(4)} ETH\n`;
  report += `🔗 Wallet: \`${shortWallet}\``;
  
  return report;
}

function formatPlainReport(data: ReportData): string {
  let report = `Moonwell Yield Report\n`;
  report += `${'='.repeat(40)}\n\n`;
  report += `Position: $${data.positionUSDC.toFixed(2)} USDC\n`;
  report += `APY: ${data.totalAPY.toFixed(2)}%\n\n`;
  
  if (data.totalRewardsUSD > 0.01) {
    report += `Pending Rewards: $${data.totalRewardsUSD.toFixed(2)}\n`;
    report += `  WELL: ${data.wellBalance.toFixed(2)}\n`;
    report += `  MORPHO: ${data.morphoBalance.toFixed(4)}\n\n`;
  }
  
  const dailyEarnings = (data.positionUSDC * data.totalAPY / 100) / 365;
  report += `Est. Daily: $${dailyEarnings.toFixed(4)}\n`;
  report += `Gas: ${data.ethBalance.toFixed(4)} ETH\n`;
  
  return report;
}

function formatJSONReport(data: ReportData): string {
  return JSON.stringify({
    ...data,
    timestamp: new Date().toISOString(),
    shouldCompound: data.totalRewardsUSD > 0.50, // Compound if >$0.50 in rewards
  }, null, 2);
}

async function main() {
  const format = process.argv.includes('--json') ? 'json'
    : process.argv.includes('--plain') ? 'plain'
    : 'telegram';
  
  const data = await gatherReportData();
  
  switch (format) {
    case 'json':
      console.log(formatJSONReport(data));
      break;
    case 'plain':
      console.log(formatPlainReport(data));
      break;
    default:
      console.log(formatTelegramReport(data));
  }
}

main().catch((err) => handleError(err, 'Report generation failed'));
