---
name: monday
description: |
  Monday.com API integration with managed OAuth. Manage boards, items, columns, groups, and workspaces using GraphQL.
  Use this skill when users want to create, update, or query Monday.com boards and items, manage tasks, or automate workflows.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
  Requires network access and valid Maton API key.
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# Monday.com

Access the Monday.com API with managed OAuth authentication. Manage boards, items, columns, groups, users, and workspaces using GraphQL.

## Quick Start

```bash
# Get current user
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'query': '{ me { id name email } }'}).encode()
req = urllib.request.Request('https://gateway.maton.ai/monday/v2', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/monday/v2
```

All requests use POST to the GraphQL endpoint. The gateway proxies requests to `api.monday.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Monday.com OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=monday&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'monday'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "ca93f2c5-5126-4360-b293-4f05f7bb6c8c",
    "status": "ACTIVE",
    "creation_time": "2026-02-05T20:10:47.585047Z",
    "last_updated_time": "2026-02-05T20:11:12.357011Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "monday",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Monday.com connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'query': '{ me { id name } }'}).encode()
req = urllib.request.Request('https://gateway.maton.ai/monday/v2', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
req.add_header('Maton-Connection', 'ca93f2c5-5126-4360-b293-4f05f7bb6c8c')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

Monday.com uses a GraphQL API. All operations are sent as POST requests with a JSON body containing the `query` field.

### Current User (me)

```bash
POST /monday/v2
Content-Type: application/json

{"query": "{ me { id name email } }"}
```

**Response:**
```json
{
  "data": {
    "me": {
      "id": "72989582",
      "name": "Chris",
      "email": "chris.kim.2332@gmail.com"
    }
  }
}
```

### Users

```bash
POST /monday/v2
Content-Type: application/json

{"query": "{ users(limit: 20) { id name email } }"}
```

### Workspaces

```bash
POST /monday/v2
Content-Type: application/json

{"query": "{ workspaces(limit: 10) { id name kind } }"}
```

**Response:**
```json
{
  "data": {
    "workspaces": [
      { "id": "10136488", "name": "Main workspace", "kind": "open" }
    ]
  }
}
```

### Boards

#### List Boards

```bash
POST /monday/v2
Content-Type: application/json

{"query": "{ boards(limit: 10) { id name state board_kind workspace { id name } } }"}
```

**Response:**
```json
{
  "data": {
    "boards": [
      {
        "id": "8614733398",
        "name": "Welcome to your developer account",
        "state": "active",
        "board_kind": "public",
        "workspace": { "id": "10136488", "name": "Main workspace" }
      }
    ]
  }
}
```

#### Get Board with Columns, Groups, and Items

```bash
POST /monday/v2
Content-Type: application/json

{"query": "{ boards(ids: [BOARD_ID]) { id name columns { id title type } groups { id title } items_page(limit: 20) { cursor items { id name state } } } }"}
```

#### Create Board

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { create_board(board_name: \"New Board\", board_kind: public) { id name } }"}
```

**Response:**
```json
{
  "data": {
    "create_board": {
      "id": "18398921201",
      "name": "New Board"
    }
  }
}
```

#### Update Board

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { update_board(board_id: BOARD_ID, board_attribute: description, new_value: \"Board description\") }"}
```

#### Delete Board

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { delete_board(board_id: BOARD_ID) { id } }"}
```

### Items

#### Get Items by ID

```bash
POST /monday/v2
Content-Type: application/json

{"query": "{ items(ids: [ITEM_ID]) { id name created_at updated_at state board { id name } group { id title } column_values { id text value } } }"}
```

**Response:**
```json
{
  "data": {
    "items": [
      {
        "id": "11200791874",
        "name": "Test item",
        "created_at": "2026-02-05T20:12:42Z",
        "updated_at": "2026-02-05T20:12:42Z",
        "state": "active",
        "board": { "id": "8614733398", "name": "Welcome to your developer account" },
        "group": { "id": "topics", "title": "Group Title" }
      }
    ]
  }
}
```

#### Create Item

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { create_item(board_id: BOARD_ID, group_id: \"GROUP_ID\", item_name: \"New item\") { id name } }"}
```

#### Create Item with Column Values

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { create_item(board_id: BOARD_ID, group_id: \"GROUP_ID\", item_name: \"New task\", column_values: \"{\\\"status\\\": {\\\"label\\\": \\\"Working on it\\\"}}\") { id name column_values { id text } } }"}
```

#### Update Item Name

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { change_simple_column_value(board_id: BOARD_ID, item_id: ITEM_ID, column_id: \"name\", value: \"Updated name\") { id name } }"}
```

#### Update Column Value

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { change_column_value(board_id: BOARD_ID, item_id: ITEM_ID, column_id: \"status\", value: \"{\\\"label\\\": \\\"Done\\\"}\") { id name } }"}
```

#### Delete Item

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { delete_item(item_id: ITEM_ID) { id } }"}
```

### Columns

#### Create Column

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { create_column(board_id: BOARD_ID, title: \"Status\", column_type: status) { id title type } }"}
```

**Response:**
```json
{
  "data": {
    "create_column": {
      "id": "color_mm09e48w",
      "title": "Status",
      "type": "status"
    }
  }
}
```

#### Column Types

Common column types: `status`, `text`, `numbers`, `date`, `people`, `dropdown`, `checkbox`, `email`, `phone`, `link`, `timeline`, `tags`, `rating`

### Groups

#### Create Group

```bash
POST /monday/v2
Content-Type: application/json

{"query": "mutation { create_group(board_id: BOARD_ID, group_name: \"New Group\") { id title } }"}
```

**Response:**
```json
{
  "data": {
    "create_group": {
      "id": "group_mm0939df",
      "title": "New Group"
    }
  }
}
```

## Pagination

Monday.com uses cursor-based pagination for items with `items_page` and `next_items_page`.

```bash
# First page
POST /monday/v2
{"query": "{ boards(ids: [BOARD_ID]) { items_page(limit: 50) { cursor items { id name } } } }"}

# Next page using cursor
POST /monday/v2
{"query": "{ next_items_page(cursor: \"CURSOR_VALUE\", limit: 50) { cursor items { id name } } }"}
```

Response includes `cursor` when more items exist (null when no more pages):

```json
{
  "data": {
    "boards": [{
      "items_page": {
        "cursor": "MSw5NzI4...",
        "items": [...]
      }
    }]
  }
}
```

## Code Examples

### JavaScript

```javascript
const response = await fetch('https://gateway.maton.ai/monday/v2', {
  method: 'POST',
  headers: {
    'Authorization': `Bearer ${process.env.MATON_API_KEY}`,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({
    query: `{ boards(limit: 10) { id name items_page(limit: 20) { items { id name } } } }`
  })
});
const data = await response.json();
```

### Python

```python
import os
import requests

response = requests.post(
    'https://gateway.maton.ai/monday/v2',
    headers={
        'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}',
        'Content-Type': 'application/json'
    },
    json={
        'query': '{ boards(limit: 10) { id name items_page(limit: 20) { items { id name } } } }'
    }
)
data = response.json()
```

## Notes

- Monday.com uses GraphQL exclusively (no REST API)
- Board IDs, item IDs, and user IDs are numeric strings
- Column IDs are alphanumeric strings (e.g., `color_mm09e48w`)
- Group IDs are alphanumeric strings (e.g., `group_mm0939df`, `topics`)
- Column values must be passed as JSON strings when creating/updating items
- The `account` query may require additional OAuth scopes. If you receive a scope error, contact Maton support at support@maton.ai with the specific operations/APIs you need and your use-case
- Board kinds: `public`, `private`, `share`
- Board states: `active`, `archived`, `deleted`, `all`
- Each cursor is valid for 60 minutes after the initial request
- Default limit is 25, maximum is 100 for most queries

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Monday.com connection or GraphQL validation error |
| 401 | Invalid or missing Maton API key |
| 403 | Insufficient OAuth scope for the operation |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Monday.com API |

GraphQL errors are returned in the `errors` array:

```json
{
  "data": {},
  "errors": [
    {
      "message": "Unauthorized field or type",
      "path": ["account"],
      "extensions": { "code": "UNAUTHORIZED_FIELD_OR_TYPE" }
    }
  ]
}
```

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `monday`. For example:

- Correct: `https://gateway.maton.ai/monday/v2`
- Incorrect: `https://gateway.maton.ai/v2`

## Resources

- [Monday.com API Basics](https://developer.monday.com/api-reference/docs/basics)
- [GraphQL Overview](https://developer.monday.com/api-reference/docs/introduction-to-graphql)
- [Boards Reference](https://developer.monday.com/api-reference/reference/boards)
- [Items Reference](https://developer.monday.com/api-reference/reference/items)
- [Columns Reference](https://developer.monday.com/api-reference/reference/columns)
- [API Changelog](https://developer.monday.com/api-reference/changelog)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
