# Moltpho API Reference

This document provides complete API documentation for the Moltpho shopping service. All endpoints use JSON request/response bodies and require HTTPS.

## Base URL

```
https://api.moltpho.com/v1
```

## Authentication

All endpoints (except registration) require authentication via one of:

1. **Bearer Token** (preferred):
   ```
   Authorization: Bearer <api_key_secret>
   ```

2. **API Key Header**:
   ```
   X-Api-Key: <api_key_secret>
   ```

3. **HMAC Signature** (recommended for production):
   ```
   X-Moltpho-Key-Id: <key_id>
   X-Moltpho-Signature: <hmac_signature>
   X-Moltpho-Timestamp: <unix_timestamp>
   ```

---

## Registration & Claim

### POST /v1/agents/register

Register a new agent account. No authentication required.

**Request:**
```json
{
  "agent_display_name": "My Shopping Agent",
  "agent_description": "Personal shopping assistant for household items",
  "openclaw_instance_id": "oc_inst_abc123"  // optional
}
```

**Response (201 Created):**
```json
{
  "agent_id": "ag_7f3a9b2c-4e5d-6f7a-8b9c-0d1e2f3a4b5c",
  "api_key_id": "moltpho_key_abc123def456",
  "api_key_secret": "moltpho_secret_xyz789...",
  "claim_url": "https://portal.moltpho.com/claim/clm_...",
  "wallet_address": "0x742d35Cc6634C0532925a3b844Bc9e7595f2bD45",
  "claim_expires_at": "2026-02-05T15:30:00Z"
}
```

**Notes:**
- `api_key_secret` is shown only once; store securely
- `claim_url` expires in 24 hours
- Credentials file should be saved with chmod 600 permissions

---

### POST /v1/agents/claim

Claim an agent account (portal endpoint). Requires valid claim token.

**Request:**
```json
{
  "claim_token": "clm_abc123...",
  "email": "owner@example.com"
}
```

**Response (200 OK):**
```json
{
  "agent_id": "ag_7f3a9b2c-4e5d-6f7a-8b9c-0d1e2f3a4b5c",
  "owner_user_id": "usr_1a2b3c4d-5e6f-7a8b-9c0d-1e2f3a4b5c6d",
  "status": "CLAIMED",
  "session_token": "sess_..."
}
```

**Errors:**
- `401 UNAUTHORIZED` - Invalid or expired claim token
- `409 CONFLICT` - Agent already claimed

---

### GET /v1/agents/me

Get current agent details.

**Response (200 OK):**
```json
{
  "agent_id": "ag_7f3a9b2c-4e5d-6f7a-8b9c-0d1e2f3a4b5c",
  "display_name": "My Shopping Agent",
  "description": "Personal shopping assistant",
  "status": "CLAIMED",
  "wallet_address": "0x742d35Cc6634C0532925a3b844Bc9e7595f2bD45",
  "created_at": "2026-02-04T10:00:00Z",
  "owner_user_id": "usr_1a2b3c4d-5e6f-7a8b-9c0d-1e2f3a4b5c6d"
}
```

---

### POST /v1/agents/{id}/regenerate_key

Regenerate API key for an agent. Old key is invalidated immediately. Portal only, requires owner scope.

**Response (200 OK):**
```json
{
  "api_key_id": "moltpho_key_newkey789",
  "api_key_secret": "moltpho_secret_newsecret...",
  "previous_key_revoked_at": "2026-02-04T12:00:00Z"
}
```

---

## Profile

### GET /v1/shipping_profiles

List shipping profiles for the authenticated agent.

**Response (200 OK):**
```json
{
  "profiles": [
    {
      "id": "sp_abc123",
      "full_name": "John Doe",
      "address1": "123 Main St",
      "address2": "Apt 4B",
      "city": "San Francisco",
      "state": "CA",
      "postal_code": "94102",
      "country": "US",
      "email": "john@example.com",
      "phone": "+1-555-123-4567",
      "is_default": true,
      "created_at": "2026-02-04T10:00:00Z",
      "updated_at": "2026-02-04T10:00:00Z"
    }
  ]
}
```

---

### POST /v1/shipping_profiles

Create or update the default shipping profile (upsert). If a default profile exists, it is updated; otherwise a new one is created.

**Request:**
```json
{
  "full_name": "John Doe",
  "address1": "123 Main St",
  "address2": "Apt 4B",
  "city": "San Francisco",
  "state": "CA",
  "postal_code": "94102",
  "country": "US",
  "email": "john@example.com",
  "phone": "+1-555-123-4567"
}
```

**Fields:**
| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `full_name` | string | Yes | Recipient full name (max 200) |
| `address1` | string | Yes | Street address line 1 (max 200) |
| `address2` | string | No | Street address line 2 (max 200) |
| `city` | string | Yes | City (max 100) |
| `state` | string | Yes | State code (max 100) |
| `postal_code` | string | Yes | ZIP/postal code (max 20) |
| `country` | string | No | Country code (default "US", only US supported) |
| `email` | string | Yes | Contact email |
| `phone` | string | Yes | Contact phone (5-30 chars) |

**Response (200 OK):**
```json
{
  "shipping_profile": {
    "id": "sp_abc123",
    "full_name": "John Doe",
    "address1": "123 Main St",
    "address2": "Apt 4B",
    "city": "San Francisco",
    "state": "CA",
    "postal_code": "94102",
    "country": "US",
    "email": "john@example.com",
    "phone": "+1-555-123-4567",
    "is_default": true,
    "created_at": "2026-02-04T10:00:00Z",
    "updated_at": "2026-02-04T12:00:00Z"
  }
}
```

**Errors:**
- `422 INVALID_SHIPPING_PROFILE` - Non-US address or invalid format

---

## Credit & Balance

### GET /v1/balance

Get current credit balance for the authenticated agent.

**Response (200 OK):**
```json
{
  "available_credit_cents": 50000,
  "available_credit_usd": "500.00",
  "staged_refunds_cents": 2500,
  "staged_refunds_usd": "25.00",
  "display_balance": "$500.00*",
  "display_note": "Includes $25.00 pending verification",
  "musd_balance_atomic": 5000000000,
  "soft_reservations_cents": 7500,
  "effective_available_cents": 42500
}
```

**Notes:**
- `available_credit_cents` includes staged refunds (shown with asterisk)
- `effective_available_cents` = available - soft_reservations
- `musd_balance_atomic` is the raw on-chain balance (6 decimals)

---

### GET /v1/credit_policy

Get credit policy configuration for the authenticated agent.

**Response (200 OK):**
```json
{
  "agent_id": "ag_7f3a9b2c-4e5d-6f7a-8b9c-0d1e2f3a4b5c",
  "target_limit_cents": 100000,
  "target_limit_usd": "1000.00",
  "currency": "USD",
  "weekly_topup_day": "SUNDAY",
  "weekly_topup_timezone": "America/Los_Angeles",
  "autonomous_purchasing_enabled": true,
  "proactive_purchasing_enabled": true,
  "confirmation_required": false,
  "per_order_cap_cents": null,
  "daily_cap_cents": null,
  "category_allowlist": null,
  "category_denylist": null,
  "card_pool_id": "cp_xyz789"
}
```

---

### PATCH /v1/credit_policy

Update credit policy. Portal only.

**Request:**
```json
{
  "target_limit_cents": 200000,
  "autonomous_purchasing_enabled": true,
  "proactive_purchasing_enabled": false,
  "per_order_cap_cents": 10000,
  "daily_cap_cents": 25000,
  "category_denylist": ["electronics", "luxury"]
}
```

**Response (200 OK):**
Returns the updated credit policy object.

**Notes:**
- Decreasing `target_limit_cents` triggers burn + refund of excess credit
- `category_allowlist` and `category_denylist` use keyword matching (see POLICIES.md)

---

## Catalog

### GET /v1/catalog/search

Search the product catalog.

**Query Parameters:**
| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `query` | string | Yes | - | Search query |
| `limit` | integer | No | 20 | Results per page (max 50) |
| `offset` | integer | No | 0 | Pagination offset |

**Example:**
```
GET /v1/catalog/search?query=wireless+mouse&limit=10&offset=0
```

**Response (200 OK):**
```json
{
  "items": [
    {
      "asin": "B08N5WRWNW",
      "title": "Logitech MX Master 3 Wireless Mouse",
      "brand": "Logitech",
      "price_cents": 10999,
      "price_usd": "109.99",
      "currency": "USD",
      "image_url": "https://images-na.ssl-images-amazon.com/...",
      "stock_status": "IN_STOCK",
      "rating": 4.7,
      "review_count": 15234,
      "prime_eligible": true,
      "cache_status": "FRESH"
    }
  ],
  "total_results": 245,
  "limit": 10,
  "offset": 0,
  "has_more": true,
  "cache_warning": null
}
```

**Notes:**
- Prices shown are Moltpho final prices (includes 10% markup)
- `cache_status`: "FRESH", "STALE" (with warning), or "EXPIRED"
- `cache_warning`: "Prices may have changed" when cache is stale

---

### GET /v1/catalog/items/{asin}

Get detailed product information.

**Response (200 OK):**
```json
{
  "asin": "B08N5WRWNW",
  "title": "Logitech MX Master 3 Wireless Mouse",
  "brand": "Logitech",
  "price_cents": 10999,
  "price_usd": "109.99",
  "currency": "USD",
  "images": [
    "https://images-na.ssl-images-amazon.com/images/I/...",
    "https://images-na.ssl-images-amazon.com/images/I/..."
  ],
  "description": "Advanced wireless mouse with...",
  "bullet_points": [
    "Ergonomic design for all-day comfort",
    "MagSpeed electromagnetic scrolling",
    "USB-C quick charging"
  ],
  "category_path": ["Electronics", "Computers", "Accessories", "Mice"],
  "stock_status": "IN_STOCK",
  "estimated_delivery": "Feb 6-8",
  "rating": 4.7,
  "review_count": 15234,
  "prime_eligible": true,
  "fetched_at": "2026-02-04T14:30:00Z",
  "cache_expires_at": "2026-02-04T15:30:00Z",
  "cache_warning": null
}
```

**Errors:**
- `404 NOT_FOUND` - ASIN does not exist
- `422 UNPROCESSABLE_ENTITY` - Item exceeds MAX_SALE_USD limit

---

## Quotes

### POST /v1/quotes

Create a price quote for an item. Creates a soft reservation against available credit.

**Headers:**
```
Idempotency-Key: idem_abc123xyz
```

**Request:**
```json
{
  "asin": "B08N5WRWNW",
  "quantity": 1,
  "shipping_profile_id": "sp_abc123"
}
```

**Response (201 Created):**
```json
{
  "quote_id": "qt_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f",
  "soft_reservation_id": "sr_abc123",
  "asin": "B08N5WRWNW",
  "quantity": 1,
  "item_price_cents": 10999,
  "tax_cents_est": 990,
  "shipping_cents_est": 0,
  "total_due_cents": 11989,
  "total_due_usd": "119.89",
  "stock_status": "IN_STOCK",
  "expires_at": "2026-02-04T15:40:00Z",
  "payment_requirements": {
    "scheme": "x402",
    "network": "eip155:8453",
    "token": "0x...",
    "recipient": "0x...",
    "amount": "119890000"
  },
  "created_at": "2026-02-04T15:30:00Z"
}
```

**Errors:**
- `409 INSUFFICIENT_CREDIT` - Not enough available balance
- `409 MAX_CONCURRENT_QUOTES` - Already have 5 active quotes
- `422 INVALID_SHIPPING_PROFILE` - Missing or non-US shipping profile

---

### GET /v1/quotes/{id}

Get quote details.

**Response (200 OK):**
Returns the quote object (same structure as POST response).

**Errors:**
- `404 NOT_FOUND` - Quote does not exist
- `409 QUOTE_EXPIRED` - Quote TTL exceeded

---

### DELETE /v1/quotes/{id}

Cancel a quote and release the soft reservation.

**Response (204 No Content)**

**Notes:**
- Releases the soft reservation immediately
- Does not affect available credit for future quotes

---

## Orders

### POST /v1/orders

Create an order. This endpoint is x402 paywalled.

**Headers:**
```
Idempotency-Key: idem_order_xyz789
```

**First Request (no payment signature):**
```json
{
  "quote_id": "qt_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f",
  "shipping_profile_id": "sp_abc123"
}
```

**Response (402 Payment Required):**
```
HTTP/1.1 402 Payment Required
PAYMENT-REQUIRED: {"scheme":"x402","network":"eip155:8453","token":"0x...","recipient":"0x...","amount":"119890000","validBefore":1707064800}
```

**Second Request (with payment signature):**
```
PAYMENT-SIGNATURE: <base64_encoded_eip3009_authorization>
```

**Response (201 Created):**
```json
{
  "order_id": "ord_abc123def456",
  "status": "PAID",
  "quote_id": "qt_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f",
  "total_cents": 11989,
  "total_usd": "119.89",
  "items": [
    {
      "asin": "B08N5WRWNW",
      "title": "Logitech MX Master 3 Wireless Mouse",
      "quantity": 1,
      "price_cents": 10999
    }
  ],
  "shipping_profile_id": "sp_abc123",
  "cancellation_window_ends_at": "2026-02-04T15:35:00Z",
  "created_at": "2026-02-04T15:30:00Z",
  "paid_at": "2026-02-04T15:30:05Z"
}
```

**Errors:**
- `402 PAYMENT_REQUIRED` - Payment signature required (x402)
- `409 PRICE_CHANGED` - Price increased >2% since quote
- `409 QUOTE_EXPIRED` - Quote TTL exceeded
- `409 QUOTE_RETRY_LIMIT_EXCEEDED` - Failed 3 auto-retry attempts
- `409 INSUFFICIENT_CREDIT` - Not enough balance
- `422 AGENT_SUSPENDED` - Agent is suspended
- `503 TOKEN_PAUSED` - System halted

---

### GET /v1/orders/{id}

Get order details.

**Response (200 OK):**
```json
{
  "order_id": "ord_abc123def456",
  "status": "PLACED",
  "quote_id": "qt_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f",
  "total_cents": 11989,
  "total_usd": "119.89",
  "original_quote_cents": 11989,
  "price_tolerance_applied_bps": null,
  "items": [
    {
      "asin": "B08N5WRWNW",
      "title": "Logitech MX Master 3 Wireless Mouse",
      "quantity": 1,
      "price_cents": 10999
    }
  ],
  "shipping_profile_id": "sp_abc123",
  "amazon_order_ref": "113-1234567-8901234",
  "tracking_ref": "1Z999AA10123456784",
  "tracking_url": "https://www.ups.com/track?tracknum=...",
  "confidence_tier": null,
  "decision_reason": null,
  "created_at": "2026-02-04T15:30:00Z",
  "paid_at": "2026-02-04T15:30:05Z",
  "placed_at": "2026-02-04T15:45:00Z",
  "cancellation_window_ends_at": null
}
```

---

### GET /v1/orders

List orders with pagination and filtering.

**Query Parameters:**
| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `limit` | integer | No | 50 | Results per page (max 50) |
| `offset` | integer | No | 0 | Pagination offset |
| `from_date` | string | No | - | ISO 8601 date filter start |
| `to_date` | string | No | - | ISO 8601 date filter end |

**Example:**
```
GET /v1/orders?limit=20&offset=0&from_date=2026-02-01T00:00:00Z
```

**Response (200 OK):**
```json
{
  "orders": [
    {
      "order_id": "ord_abc123def456",
      "status": "PLACED",
      "total_cents": 11989,
      "total_usd": "119.89",
      "item_count": 1,
      "created_at": "2026-02-04T15:30:00Z"
    }
  ],
  "total_count": 47,
  "limit": 20,
  "offset": 0,
  "has_more": true
}
```

---

### POST /v1/orders/{id}/cancel

Cancel an order. Only allowed before PLACED status or within 5 minutes of PAID.

**Response (200 OK):**
```json
{
  "order_id": "ord_abc123def456",
  "status": "CANCELED",
  "refund_status": "COMPLETED",
  "refund_amount_cents": 11989,
  "canceled_at": "2026-02-04T15:32:00Z"
}
```

**Errors:**
- `409 CONFLICT` - Order cannot be canceled (past cancellation window or already shipped)

---

## x402 Signing

### POST /v1/wallets/x402/sign

Request an x402 payment signature for an order. Custodial endpoint.

**Headers:**
```
Idempotency-Key: idem_sign_abc123
```

**Request:**
```json
{
  "payment_required": {
    "scheme": "x402",
    "network": "eip155:8453",
    "token": "0x...",
    "recipient": "0x...",
    "amount": "119890000",
    "validBefore": 1707064800
  },
  "quote_id": "qt_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f"
}
```

**Response (200 OK):**
```json
{
  "payment_signature": "eyJhbGciOiJFUzI1NiIsInR5cCI6IkpXVCJ9...",
  "valid_until": "2026-02-04T15:40:00Z",
  "nonce": "0x7f3a9b2c4e5d6f7a8b9c0d1e2f3a4b5c"
}
```

**Validation Rules:**
- `recipient` MUST equal MoltphoMall address
- `amount` MUST match the active quote total
- Rate limit: max 10 sign requests per minute per agent

**Errors:**
- `400 BAD_REQUEST` - Invalid payment_required format
- `403 FORBIDDEN` - Quote binding failed (amount mismatch)
- `429 RATE_LIMITED` - Exceeded 10 signs/minute

---

### GET /v1/wallets/musd_balance

Get on-chain mUSD balance for the agent wallet.

**Response (200 OK):**
```json
{
  "wallet_address": "0x742d35Cc6634C0532925a3b844Bc9e7595f2bD45",
  "musd_balance_atomic": 5000000000,
  "musd_balance_usd": "500.00",
  "last_synced_at": "2026-02-04T15:30:00Z"
}
```

**Notes:**
- On-chain balance may temporarily differ from credit balance during settlement
- `musd_balance_atomic` uses 6 decimals (divide by 10^6 for USD)

---

## Support Tickets

### POST /v1/support_tickets

Create a new support ticket.

**Request:**
```json
{
  "type": "RETURN",
  "order_id": "ord_abc123def456",
  "description": "Item arrived damaged, requesting return"
}
```

**Fields:**
| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `type` | enum | Yes | `RETURN`, `LOST_PACKAGE`, or `OTHER` |
| `order_id` | UUID | Conditional | Required for `RETURN` and `LOST_PACKAGE` types |
| `description` | string | Yes | 1-2000 characters describing the issue |

**Response (201 Created):**
```json
{
  "id": "st_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f",
  "type": "RETURN",
  "status": "OPEN",
  "created_at": "2026-02-06T10:00:00Z"
}
```

**Errors:**
- `400 BAD_REQUEST` - Invalid type, missing required order_id, or description too short/long
- `403 FORBIDDEN` - Order does not belong to this agent
- `404 NOT_FOUND` - Order not found

---

### GET /v1/support_tickets

List support tickets for the authenticated agent.

**Query Parameters:**
| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `limit` | integer | No | 50 | Results per page (max 100) |
| `offset` | integer | No | 0 | Pagination offset |

**Response (200 OK):**
```json
{
  "tickets": [
    {
      "id": "st_9f8e7d6c-5b4a-3c2d-1e0f-9a8b7c6d5e4f",
      "type": "RETURN",
      "status": "OPEN",
      "order_id": "ord_abc123def456",
      "created_at": "2026-02-06T10:00:00Z",
      "updated_at": "2026-02-06T10:00:00Z"
    }
  ],
  "total": 1,
  "limit": 50,
  "offset": 0
}
```

**Ticket Statuses:**
| Status | Description |
|--------|-------------|
| `OPEN` | Newly created, awaiting support |
| `IN_PROGRESS` | Being handled by support team |
| `WAITING_CUSTOMER` | Support is awaiting your response |
| `RESOLVED` | Issue resolved |
| `CLOSED` | Ticket closed (terminal state) |

---

## Error Codes

All error responses follow this format:

```json
{
  "error": "ERROR_CODE",
  "message": "Human-readable error description",
  "details": {}
}
```

### HTTP Status Codes

| Code | Error | Description |
|------|-------|-------------|
| 400 | `BAD_REQUEST` | Malformed request body or parameters |
| 401 | `UNAUTHORIZED` | Missing or invalid credentials |
| 402 | `PAYMENT_REQUIRED` | x402 payment signature required |
| 403 | `FORBIDDEN_SCOPE` | Valid auth but insufficient scope |
| 404 | `NOT_FOUND` | Resource does not exist |
| 409 | `PRICE_CHANGED` | Price increased >2% since quote |
| 409 | `INSUFFICIENT_CREDIT` | Not enough available balance |
| 409 | `QUOTE_EXPIRED` | Quote TTL (10 minutes) exceeded |
| 409 | `QUOTE_RETRY_LIMIT_EXCEEDED` | Failed 3 auto-retry attempts |
| 409 | `MAX_CONCURRENT_QUOTES` | Already have 5 active quotes |
| 422 | `INVALID_SHIPPING_PROFILE` | Missing or non-US address |
| 422 | `AGENT_SUSPENDED` | Agent is suspended |
| 422 | `AGENT_DEGRADED` | Agent degraded with insufficient balance |
| 429 | `RATE_LIMITED` | Too many requests |
| 503 | `PROCUREMENT_UNAVAILABLE` | Procurement service down |
| 503 | `KEY_SERVICE_UNAVAILABLE` | Order queued for later |
| 503 | `TOKEN_PAUSED` | System halted due to token pause |
| 503 | `FACILITATOR_UNAVAILABLE` | x402 facilitator unreachable |

### Error Examples

**Insufficient Credit:**
```json
{
  "error": "INSUFFICIENT_CREDIT",
  "message": "Available balance ($45.00) is less than order total ($119.89)",
  "details": {
    "available_cents": 4500,
    "required_cents": 11989
  }
}
```

**Rate Limited:**
```json
{
  "error": "RATE_LIMITED",
  "message": "Too many requests",
  "retry_after_seconds": 60
}
```

Response Headers:
```
Retry-After: 60
X-RateLimit-Limit: 20
X-RateLimit-Remaining: 0
X-RateLimit-Reset: 1707064800
```

**Price Changed:**
```json
{
  "error": "PRICE_CHANGED",
  "message": "Price increased by 5.2% since quote (exceeds 2% tolerance)",
  "details": {
    "original_price_cents": 10999,
    "current_price_cents": 11570,
    "change_bps": 520,
    "tolerance_bps": 200
  }
}
```

**Agent Suspended:**
```json
{
  "error": "AGENT_SUSPENDED",
  "message": "Agent is suspended. Contact support to resolve.",
  "details": {
    "agent_id": "ag_7f3a9b2c-4e5d-6f7a-8b9c-0d1e2f3a4b5c",
    "suspended_at": "2026-02-03T10:00:00Z"
  }
}
```

---

## Rate Limits

| Endpoint Category | Limit | Window |
|-------------------|-------|--------|
| Catalog search | 60 requests | per minute |
| Quotes | 20 requests | per minute |
| Orders | 5 requests | per minute |
| Signing | 10 requests | per minute |

Rate limit headers are included in all responses:
```
X-RateLimit-Limit: 60
X-RateLimit-Remaining: 45
X-RateLimit-Reset: 1707064800
```

---

## Idempotency

The following endpoints require an `Idempotency-Key` header:

- `POST /v1/quotes`
- `POST /v1/orders`
- `POST /v1/wallets/x402/sign`

Keys must be unique per (agent_id, endpoint) combination. Duplicate requests return the cached response from the original request.

**Example:**
```
Idempotency-Key: idem_order_20260204_abc123
```

**Notes:**
- Keys are stored for 24 hours
- Using the same key with different request bodies returns an error
- Keys should be generated client-side (UUIDs recommended)

---

## Versioning

- API version is specified in the URL path: `/v1/`, `/v2/`, etc.
- Breaking changes require a version bump
- Old versions are deprecated with 6+ month notice
- Deprecation warnings are returned in the `X-API-Deprecation` header
