#!/usr/bin/env python3
"""
MoltFlow WhatsApp Channels — Python utility script
Manage WhatsApp Channel broadcasting, scheduling, and follower sync.

Usage:
    python channels.py list
    python channels.py get <channel_id>
    python channels.py create <name> <session_id>
    python channels.py delete <channel_id>
    python channels.py broadcast <channel_id> <text>
    python channels.py schedule <channel_id> <session_id> <text> <iso_datetime>
    python channels.py sync <channel_id>
    python channels.py capabilities
    python channels.py discover <session_id>
    python channels.py import <session_id> <wa_channel_id>

Required environment:
    MOLTFLOW_API_KEY  — API key with channels:read / channels:manage scope
    MOLTFLOW_API_URL  — override base URL (default: https://apiv2.waiflow.app)
"""

import os
import sys
import json
import urllib.request
import urllib.error
from datetime import datetime, timezone

BASE_URL = os.getenv("MOLTFLOW_API_URL", "https://apiv2.waiflow.app").rstrip("/")
API_KEY = os.getenv("MOLTFLOW_API_KEY", "")


def _request(method: str, path: str, body=None):
    """Make an authenticated API request. Raises on HTTP errors."""
    url = f"{BASE_URL}/api/v2{path}"
    data = json.dumps(body).encode("utf-8") if body is not None else None
    req = urllib.request.Request(
        url,
        data=data,
        method=method,
        headers={
            "X-API-Key": API_KEY,
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
    )
    try:
        with urllib.request.urlopen(req) as resp:
            raw = resp.read()
            return json.loads(raw) if raw else {}
    except urllib.error.HTTPError as e:
        body_text = e.read().decode("utf-8", errors="replace")
        print(f"HTTP {e.code} {e.reason}: {body_text}", file=sys.stderr)
        sys.exit(1)


def list_channels():
    """List all channels for this account."""
    return _request("GET", "/channels")


def get_channel(channel_id: str):
    """Get channel details including follower count and recent posts."""
    return _request("GET", f"/channels/{channel_id}")


def create_channel(name: str, session_id: str, description: str = ""):
    """Create a new WhatsApp Channel."""
    payload = {"name": name, "session_id": session_id}
    if description:
        payload["description"] = description
    return _request("POST", "/channels", payload)


def delete_channel(channel_id: str):
    """Delete a channel (must be owner)."""
    return _request("DELETE", f"/channels/{channel_id}")


def broadcast_channel_post(channel_id: str, text: str):
    """Send an immediate text post to a channel."""
    return _request("POST", f"/channels/{channel_id}/broadcast", {"text": text})


def schedule_channel_post(
    channel_id: str,
    session_id: str,
    name: str,
    text: str,
    scheduled_time: str,
    cron_expression: str = "",
):
    """Schedule a channel post (one-time ISO 8601 datetime or recurring cron)."""
    payload = {
        "name": name,
        "session_id": session_id,
        "channel_id": channel_id,
        "target_type": "channel",
        "message_content": text,
    }
    if cron_expression:
        payload["schedule_type"] = "recurring"
        payload["cron_expression"] = cron_expression
    else:
        payload["schedule_type"] = "once"
        payload["scheduled_time"] = scheduled_time
    return _request("POST", "/scheduled-messages", payload)


def sync_channel_followers(channel_id: str):
    """Trigger on-demand follower count refresh from WAHA."""
    return _request("POST", f"/channels/{channel_id}/sync-followers")


def get_capabilities():
    """Check WAHA version compatibility for media broadcasts."""
    return _request("GET", "/channels/capabilities")


def discover_channels(session_id: str):
    """Discover existing WhatsApp channels available for import."""
    return _request("GET", f"/channels/discover?session_id={session_id}")


def import_channel(session_id: str, wa_channel_id: str):
    """Import an existing WhatsApp channel into MoltFlow."""
    return _request("POST", "/channels/import", {
        "session_id": session_id,
        "wa_channel_id": wa_channel_id,
    })


def _print(data):
    print(json.dumps(data, indent=2, default=str))


def main():
    if not API_KEY:
        print("Error: MOLTFLOW_API_KEY environment variable not set.", file=sys.stderr)
        sys.exit(1)

    args = sys.argv[1:]
    if not args:
        print(__doc__)
        sys.exit(0)

    cmd = args[0]

    if cmd == "list":
        _print(list_channels())
    elif cmd == "get":
        if len(args) < 2:
            print("Usage: channels.py get <channel_id>", file=sys.stderr)
            sys.exit(1)
        _print(get_channel(args[1]))
    elif cmd == "create":
        if len(args) < 3:
            print("Usage: channels.py create <name> <session_id> [description]", file=sys.stderr)
            sys.exit(1)
        desc = args[3] if len(args) > 3 else ""
        _print(create_channel(args[1], args[2], desc))
    elif cmd == "delete":
        if len(args) < 2:
            print("Usage: channels.py delete <channel_id>", file=sys.stderr)
            sys.exit(1)
        _print(delete_channel(args[1]))
    elif cmd == "broadcast":
        if len(args) < 3:
            print("Usage: channels.py broadcast <channel_id> <text>", file=sys.stderr)
            sys.exit(1)
        _print(broadcast_channel_post(args[1], args[2]))
    elif cmd == "schedule":
        if len(args) < 5:
            print("Usage: channels.py schedule <channel_id> <session_id> <text> <iso_datetime>", file=sys.stderr)
            sys.exit(1)
        _print(schedule_channel_post(args[1], args[2], "Scheduled Post", args[3], args[4]))
    elif cmd == "sync":
        if len(args) < 2:
            print("Usage: channels.py sync <channel_id>", file=sys.stderr)
            sys.exit(1)
        _print(sync_channel_followers(args[1]))
    elif cmd == "capabilities":
        _print(get_capabilities())
    elif cmd == "discover":
        if len(args) < 2:
            print("Usage: channels.py discover <session_id>", file=sys.stderr)
            sys.exit(1)
        _print(discover_channels(args[1]))
    elif cmd == "import":
        if len(args) < 3:
            print("Usage: channels.py import <session_id> <wa_channel_id>", file=sys.stderr)
            sys.exit(1)
        _print(import_channel(args[1], args[2]))
    else:
        print(f"Unknown command: {cmd}", file=sys.stderr)
        print("Available: list, get, create, delete, broadcast, schedule, sync, capabilities, discover, import")
        sys.exit(1)


if __name__ == "__main__":
    main()
