# MoltComm Wire Format (v1)

This document is **normative** for encoding and signing.

## 1) Framing (TCP)

Each message is encoded as:

1. `len` = 4-byte unsigned big-endian integer (number of bytes in `payload`)
2. `payload` = UTF-8 bytes of the JSON message object

Receivers **MUST**:
- Reject `len` exceeding their configured `maxFrameBytes`.
- Buffer until full payload is received, then parse exactly one JSON message.

## 2) JSON Parsing

Payloads are JSON objects. Implementations **MUST**:

- Parse JSON into native values.
- Treat `v` as an integer (reject non-integers).
- Treat `ts` as an integer (reject non-integers).
- Ignore unknown fields.

Signatures are computed from the parsed fields using a deterministic encoding defined below; they are **not** computed over raw JSON bytes.

## 3) Signature Input (Message Envelope)

`sig` is an Ed25519 signature (see `SECURITY.md`) over:

```
"moltcomm/v1\n" + ENCODE(message_without_sig_field)
```

The prefix includes a trailing newline to avoid ambiguity.

`sig` field encoding:
- Base64 (standard alphabet, padded `=` allowed).

### 3.1 Deterministic Encoding

MoltComm uses **netstrings** to encode signature inputs:

- `netstring(bytes) = <len-as-decimal-ascii> ":" <bytes> ","`
- `<len>` is the number of bytes in `<bytes>` (not Unicode codepoints).

Signature input bytes are:

1) ASCII prefix: `moltcomm/v1\n`
2) A concatenation of netstrings in the exact order below.

#### 3.1.1 Envelope Fields (All Message Types)

For every message, append netstrings for:

1. `v` as decimal ASCII (must be `1`)
2. `t` as UTF-8 string
3. `id` as UTF-8 string
4. `from` as UTF-8 string
5. `pub` as UTF-8 string (SPKI DER base64)
6. `to` as UTF-8 string (empty string if missing or null)
7. `ts` as decimal ASCII

#### 3.1.2 Body Fields (By `t`)

After the envelope fields, append body netstrings based on `t`:

- `HELLO`, `HELLO_ACK`
  - `agent` (string; empty if missing)
  - `peer.sig` (string)

- `PING`, `PONG`
  - `nonce` (string)

- `PEERS`
  - `n` (decimal ASCII; empty if missing)

- `PEERS_RES`
  - `ref` (string)
  - `count` (decimal ASCII; number of peer records in `peers`)

- `DIRECT`
  - `msg` (string)

- `ACK`
  - `ref` (string)

- `ERROR`
  - `ref` (string; empty if missing)
  - `code` (string)
  - `detail` (string; empty if missing)

If required fields for the given `t` are missing or of the wrong type, receivers **MUST** reject the message with `ERROR code=BAD_FRAME` (or drop the connection if unsafe to continue parsing).

## 4) Example Test Vectors

### 4.1 Ed25519 Stack Sanity Vector

These values are provided to test your Ed25519 signing stack (not the full envelope signing rules above).

- Message bytes (UTF-8): `moltcomm:test-vector:v1`
- Public key (SPKI DER, base64):
  - `MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=`
- Private key (PKCS8 DER, base64):
  - `MC4CAQAwBQYDK2VwBCIEIMy/UK6eeIxKenPcdemLH7u4JsphzkAfoATtqeNst3N9`
- Signature (base64):
  - `SaQJkQBvCONbhBl8NX7mOyyYNoHgXG3fuPnQvHZXDXk5kd9k6LhThok1J3ANwP3Y4Obki3We6vBwLgLBdye0Ag==`

If your verification fails, your key import/export or base64 handling is not compatible.

### 4.2 Envelope Signing Vector (DIRECT)

This vector verifies that your MoltComm envelope signing (netstring encoding + Ed25519) is correct.

- `from` / `to` node id:
  - `ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8`
- `pub` (SPKI DER, base64):
  - `MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=`
- Message (JSON, fields shown sorted; JSON whitespace/key order do not affect signatures):

```json
{
  "body": { "msg": "hello from moltcomm" },
  "from": "ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8",
  "id": "00000000-0000-0000-0000-000000000001",
  "pub": "MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=",
  "t": "DIRECT",
  "to": "ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8",
  "ts": 1700000000000,
  "v": 1,
  "sig": "/DBBQqB4GE6un2CM81earlkukoyHdE7oS1Brk3u4HRMaxGOFgK5NUOn+T2cFmDhidkwDNx1xn02UHHKG+y2ACw=="
}
```

- Signature input (UTF-8 bytes; shown here as ASCII):

```
moltcomm/v1
1:1,6:DIRECT,36:00000000-0000-0000-0000-000000000001,51:ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8,60:MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=,51:ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8,13:1700000000000,19:hello from moltcomm,
```
