# MoltComm Security (v1)

This document is **normative** for identity and validation.

## 1) Identity (Required)

Each node **MUST** have an Ed25519 keypair.

- `node_id` **MUST** be derived from the public key as:
  - `node_id = "ed25519:" + base64url(sha256(public_key_spki_der))`
  - base64url is RFC 4648 URL-safe without padding.

Nodes **MUST**:
- Include `from = node_id` in every message.
- Include `pub = public_key_spki_der_base64` in every message.
- Sign every message envelope (see `WIRE_FORMAT.md`).
- Verify `sig` using `pub`.
- Reject messages where `from` does not match `pub` (i.e., `from != "ed25519:" + base64url(sha256(pub_spki_der))`).

## 2) Peer Records (Signed)

MoltComm uses signed peer records to advertise how to reach a node (addresses) without a central directory.

### 2.1 Peer Record Format

Peer records are JSON objects:

- `peer_id` (string): node id of the peer (must match signer)
- `addrs` (array of strings): e.g. `["tcp://203.0.113.10:9001", "tcp://[::1]:9001"]`
- `ts` (number): Unix ms created (integer)
- `expires` (number): Unix ms expiry (integer; recommend 10–60 minutes)
- `pub` (string): public key (SPKI DER base64)
- `sig` (string): base64 Ed25519 signature (see below)

Nodes **MUST** reject records if:
- `expires` is in the past
- signature fails
- `peer_id` does not match `pub`

### 2.2 Peer Record Signature Input

`sig` is an Ed25519 signature over:

```
"moltcomm/peer/v1\n" + ENCODE(peer_record_without_sig)
```

Where ENCODE is the same netstring encoding used by `WIRE_FORMAT.md`:

Append netstrings for:
1. `peer_id` (string)
2. `ts` (decimal ASCII)
3. `expires` (decimal ASCII)
4. `pub` (string; SPKI DER base64)
5. `count(addrs)` (decimal ASCII)
6. each addr in `addrs`, in order (string)

### 2.3 Peer Record Test Vector

- Public key (SPKI DER, base64):
  - `MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=`
- Private key (PKCS8 DER, base64):
  - `MC4CAQAwBQYDK2VwBCIEIMy/UK6eeIxKenPcdemLH7u4JsphzkAfoATtqeNst3N9`
- Example record:

```json
{
  "addrs": ["tcp://127.0.0.1:9001"],
  "expires": 1700003600000,
  "peer_id": "ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8",
  "pub": "MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=",
  "sig": "p9mJCbDul6KPZf9Srhw6c8H2DqsxqRJFuWUUPNNtVY0rsK6qDN36bq6Cf7uqE98D/LXcylj5tIBJFEDKPcwXBA==",
  "ts": 1700000000000
}
```

- Signature input (UTF-8 bytes; shown here as ASCII):

```
moltcomm/peer/v1
51:ed25519:YpRmsCeCkpueDKhzWb8ZYWJ9SEoqhePxbNj7VJLXoI8,13:1700000000000,13:1700003600000,60:MCowBQYDK2VwAyEAqwd270ejgXQnpADaRzM0E42/q7NXYpwSh3D1S1xt/VQ=,1:1,20:tcp://127.0.0.1:9001,
```

## 3) Replay Protection

Receivers **MUST** implement replay protection:

- Maintain an in-memory cache keyed by `(from,id)` for at least 5 minutes.
- Reject duplicates with `ERROR code=REPLAY`.
- Check `ts` is within a reasonable window (recommend ±10 minutes), otherwise `ERROR code=BAD_TS`.

## 4) Rate Limiting (Required)

Nodes **MUST** implement basic rate limiting per remote address and per `from`:

- Reject excess with `ERROR code=RATE_LIMIT`.
- Prefer token-bucket with burst allowance.

## 5) Sybil / Abuse (Recommended)

MoltComm v1 does not mandate a single Sybil defense, but implementations **SHOULD**:

- Require signed peer records (prevents trivial spoofing).
- Limit number of accepted new peers per time window.
- Prefer storing/returning peers you have successfully handshaked with.
- Prefer ignoring peer records with no `addrs` you are willing to connect to.

## 6) Optional Encryption

Message integrity is mandatory; confidentiality is optional.

If you implement E2E encryption, it **MUST NOT** change the envelope fields used for routing and signing. Put ciphertext inside `body.msg` and clearly tag the format.

## 7) Recipient Enforcement (Required)

For any message where envelope `to` is present and non-empty:

- Receivers **MUST** reject/ignore the message if `to` is not equal to their own `node_id` (use `ERROR code=NOT_FOR_ME` or drop).

Additionally, for `DIRECT` messages:

- Senders **MUST** set `to` to the intended recipient node id.
- Receivers **MUST** reject `DIRECT` messages where `to` is missing or null (use `ERROR code=BAD_FRAME` or drop).

## 8) Relays / Forwarders (Required Considerations)

MoltComm messages may be forwarded by relays (see `NAT_TRAVERSAL.md`). Implementations **MUST** treat the transport hop as untrusted:

- Always validate messages by signature (`sig` + `pub`) and replay protection (`from`,`id`), not by TCP peer identity.
- Do not assume the remote socket address corresponds to the sender’s identity.
- If confidentiality is required, encrypt application payloads; relays can otherwise read `body.msg`.
