import { describe, expect, it } from "vitest";
import { buildEmailBody, buildEmailSubject, parseTask } from "../src/task-parser.js";

// 固定参考时间：2026-02-04 周三 10:00
const REF = new Date(2026, 1, 4, 10, 0, 0);

// === 中文解析（原有测试） ===

describe("parseTask - 中文", () => {
    it("应解析基本任务", () => {
        const result = parseTask("添加任务：买牛奶");
        expect(result).toMatchObject({ title: "买牛奶" });
    });

    it("应解析不同前缀的任务", () => {
        expect(parseTask("创建待办：写报告").title).toBe("写报告");
        expect(parseTask("提醒我：打电话").title).toBe("打电话");
        expect(parseTask("记录任务：开会").title).toBe("开会");
    });

    it("应解析无前缀的输入", () => {
        const result = parseTask("买东西");
        expect(result.title).toBe("买东西");
    });

    it("应提取列表信息 - '到X列表' 格式", () => {
        const result = parseTask("添加任务到工作列表：完成项目报告");
        expect(result.title).toBe("完成项目报告");
        expect(result.list).toBe("工作");
    });

    it("应提取列表信息 - '列表是X' 格式", () => {
        const result = parseTask("添加任务：完成报告，列表是工作");
        expect(result.title).toBe("完成报告");
        expect(result.list).toBe("工作");
    });

    it("应提取标签信息 - '标签是X和Y' 格式", () => {
        const result = parseTask("添加任务：买菜，标签是家务和购物");
        expect(result.title).toBe("买菜");
        expect(result.tags).toEqual(["家务", "购物"]);
    });

    it("应同时提取列表和标签", () => {
        const result = parseTask("添加任务：完成季度报告，列表是工作，标签是紧急和财务");
        expect(result.title).toBe("完成季度报告");
        expect(result.list).toBe("工作");
        expect(result.tags).toEqual(["紧急", "财务"]);
    });

    it("应抛出错误当标题为空时", () => {
        expect(() => parseTask("添加任务：")).toThrow("无法解析任务标题");
    });
});

// === 英文解析（Phase 2 新增） ===

describe("parseTask - 英文前缀", () => {
    it("应解析 'add task:'", () => {
        const result = parseTask("add task: buy milk");
        expect(result.title).toBe("buy milk");
    });

    it("应解析 'create todo:'", () => {
        const result = parseTask("create todo: write report");
        expect(result.title).toBe("write report");
    });

    it("应解析 'remind me to'（无冒号）", () => {
        const result = parseTask("remind me to call John");
        expect(result.title).toBe("call John");
    });

    it("应解析 'remember to'", () => {
        const result = parseTask("remember to submit the form");
        expect(result.title).toBe("submit the form");
    });

    it("应解析英文前缀含列表 'add task to work list:'", () => {
        const result = parseTask("add task to work list: finish report");
        expect(result.title).toBe("finish report");
        expect(result.list).toBe("work");
    });

    it("应解析英文列表 ', list X'", () => {
        const result = parseTask("add task: finish report, list work");
        expect(result.title).toBe("finish report");
        expect(result.list).toBe("work");
    });

    it("应解析英文标签 ', tag X and Y'", () => {
        const result = parseTask("add task: buy groceries, tag food and shopping");
        expect(result.title).toBe("buy groceries");
        expect(result.tags).toEqual(["food", "shopping"]);
    });

    it("应解析英文标签 ', tags X, Y'（不区分大小写）", () => {
        const result = parseTask("Add Task: deploy, Tags urgent finance");
        expect(result.title).toBe("deploy");
        expect(result.tags).toEqual(["urgent", "finance"]);
    });
});

// === 优先级提取（Phase 2 新增） ===

describe("parseTask - 优先级", () => {
    it("应提取 '紧急' 为高优先级", () => {
        const result = parseTask("添加任务：完成报告，紧急");
        expect(result.title).toBe("完成报告");
        expect(result.priority).toBe("high");
    });

    it("应提取 '很紧急' 为高优先级", () => {
        const result = parseTask("添加任务：修复 bug，很紧急");
        expect(result.title).toBe("修复 bug");
        expect(result.priority).toBe("high");
    });

    it("应提取 '重要' 为中优先级", () => {
        const result = parseTask("添加任务：写文档，重要");
        expect(result.title).toBe("写文档");
        expect(result.priority).toBe("medium");
    });

    it("应提取 '很重要' 为高优先级", () => {
        const result = parseTask("添加任务：季度汇报，很重要");
        expect(result.title).toBe("季度汇报");
        expect(result.priority).toBe("high");
    });

    it("应提取 '不急' 为低优先级", () => {
        const result = parseTask("添加任务：整理文件，不急");
        expect(result.title).toBe("整理文件");
        expect(result.priority).toBe("low");
    });

    it("应提取英文 'urgent'", () => {
        const result = parseTask("add task: fix production bug, urgent");
        expect(result.title).toBe("fix production bug");
        expect(result.priority).toBe("high");
    });

    it("应提取英文 'important'", () => {
        const result = parseTask("add task: review PR, important");
        expect(result.title).toBe("review PR");
        expect(result.priority).toBe("medium");
    });

    it("应同时提取优先级和标签", () => {
        const result = parseTask("添加任务：完成报告，紧急，标签是工作和财务");
        expect(result.title).toBe("完成报告");
        expect(result.priority).toBe("high");
        expect(result.tags).toEqual(["工作", "财务"]);
    });
});

// === 日期时间提取（Phase 2 新增） ===

describe("parseTask - 日期时间", () => {
    it("应提取 '明天' 并设置 dueDate", () => {
        const result = parseTask("添加任务：明天去超市买菜", REF);
        expect(result.title).toBe("去超市买菜");
        expect(result.dueDate).toBeDefined();
        expect(result.dueDate!.getDate()).toBe(5);
    });

    it("应提取 '下午3点' 并设置时间", () => {
        const result = parseTask("添加任务：下午3点开会", REF);
        expect(result.title).toBe("开会");
        expect(result.dueDate!.getHours()).toBe(15);
    });

    it("应提取 '明天下午2点' 组合日期时间", () => {
        const result = parseTask("提醒我：明天下午2点开会", REF);
        expect(result.title).toBe("开会");
        expect(result.dueDate!.getDate()).toBe(5);
        expect(result.dueDate!.getHours()).toBe(14);
    });

    it("应提取 '周五前' 并清理 '前' 字", () => {
        const result = parseTask("添加任务：周五前交报告", REF);
        expect(result.title).toBe("交报告");
        expect(result.dueDate!.getDate()).toBe(6);
    });

    it("无日期信息时 dueDate 为 undefined", () => {
        const result = parseTask("添加任务：买牛奶");
        expect(result.dueDate).toBeUndefined();
    });

    it("应同时提取日期、列表和标签", () => {
        const result = parseTask("添加任务：明天完成报告，列表是工作，标签是紧急", REF);
        expect(result.title).toBe("完成报告");
        expect(result.dueDate!.getDate()).toBe(5);
        expect(result.list).toBe("工作");
        expect(result.tags).toEqual(["紧急"]);
    });
});

// === buildEmailSubject（保持原有测试） ===

describe("buildEmailSubject", () => {
    it("应构造纯标题主题", () => {
        const subject = buildEmailSubject({ title: "买牛奶" });
        expect(subject).toBe("买牛奶");
    });

    it("应包含列表信息", () => {
        const subject = buildEmailSubject({ title: "买牛奶", list: "购物" });
        expect(subject).toBe("买牛奶 list(购物)");
    });

    it("应包含标签信息", () => {
        const subject = buildEmailSubject({ title: "买牛奶", tags: ["家务", "购物"] });
        expect(subject).toBe("买牛奶 tag(家务, 购物)");
    });

    it("应同时包含列表和标签", () => {
        const subject = buildEmailSubject({
            title: "完成报告",
            list: "工作",
            tags: ["紧急", "财务"],
        });
        expect(subject).toBe("完成报告 list(工作) tag(紧急, 财务)");
    });

    it("应包含仅日期的 due()", () => {
        const dueDate = new Date(2026, 1, 5, 0, 0);
        const subject = buildEmailSubject({ title: "买菜", dueDate });
        expect(subject).toBe("买菜 due(2-5-2026)");
    });

    it("应包含带时间的 start() 和 due()", () => {
        const dueDate = new Date(2026, 1, 5, 15, 0);
        const subject = buildEmailSubject({ title: "开会", dueDate });
        expect(subject).toBe("开会 start(2-5-2026 3pm) due(2-5-2026 3pm)");
    });

    it("应包含带分钟的 start() 和 due()", () => {
        const dueDate = new Date(2026, 1, 5, 15, 30);
        const subject = buildEmailSubject({ title: "开会", dueDate });
        expect(subject).toBe("开会 start(2-5-2026 3:30pm) due(2-5-2026 3:30pm)");
    });

    it("应同时包含日期、列表和标签", () => {
        const dueDate = new Date(2026, 1, 5, 14, 0);
        const subject = buildEmailSubject({
            title: "完成报告",
            dueDate,
            list: "工作",
            tags: ["紧急"],
        });
        expect(subject).toBe("完成报告 start(2-5-2026 2pm) due(2-5-2026 2pm) list(工作) tag(紧急)");
    });

    it("应支持 titlePrefix", () => {
        const subject = buildEmailSubject({ title: "开会" }, "2Do:");
        expect(subject).toBe("2Do:开会");
    });
});

// === buildEmailBody ===

describe("buildEmailBody", () => {
    it("应包含创建时间和来源", () => {
        const body = buildEmailBody({ title: "测试任务" });
        expect(body).toContain("创建时间：");
        expect(body).toContain("来源：Moltbot 2Do Task Email Skill");
    });

    it("应包含原始输入", () => {
        const body = buildEmailBody({ title: "买牛奶" }, "添加任务：买牛奶");
        expect(body).toContain("任务详情：添加任务：买牛奶");
    });

    it("应包含截止时间", () => {
        const dueDate = new Date(2026, 1, 5, 15, 0);
        const body = buildEmailBody({ title: "开会", dueDate });
        expect(body).toContain("截止时间：2026-02-05 15:00");
    });

    it("应包含优先级信息", () => {
        const body = buildEmailBody({ title: "报告", priority: "high" });
        expect(body).toContain("优先级：高");
    });

    it("仅日期无时间时不显示时间部分", () => {
        const dueDate = new Date(2026, 1, 5, 0, 0);
        const body = buildEmailBody({ title: "买菜", dueDate });
        expect(body).toContain("截止时间：2026-02-05");
        expect(body).not.toContain("00:00");
    });
});
