import { describe, expect, it } from "vitest";
import { extractDateTime, format2DoDate, formatDueDate } from "../src/date-parser.js";

// 固定参考时间：2026-02-04 周三 10:00
const REF = new Date(2026, 1, 4, 10, 0, 0);

describe("extractDateTime", () => {
    describe("相对日期", () => {
        it("应解析 '今天'", () => {
            const r = extractDateTime("今天开会", REF)!;
            expect(r.dueDate.getDate()).toBe(4);
            expect(r.dueDate.getMonth()).toBe(1);
            expect(r.cleanText).toBe("开会");
        });

        it("应解析 '明天'", () => {
            const r = extractDateTime("明天去超市", REF)!;
            expect(r.dueDate.getDate()).toBe(5);
            expect(r.cleanText).toBe("去超市");
        });

        it("应解析 '后天'", () => {
            const r = extractDateTime("后天交报告", REF)!;
            expect(r.dueDate.getDate()).toBe(6);
            expect(r.cleanText).toBe("交报告");
        });

        it("应解析 '大后天'", () => {
            const r = extractDateTime("大后天出发", REF)!;
            expect(r.dueDate.getDate()).toBe(7);
            expect(r.cleanText).toBe("出发");
        });
    });

    describe("星期几", () => {
        it("应解析 '周五'（本周）", () => {
            // REF 是周三，周五 = +2 天
            const r = extractDateTime("周五开会", REF)!;
            expect(r.dueDate.getDate()).toBe(6);
            expect(r.cleanText).toBe("开会");
        });

        it("应解析 '星期一'（下一个周一）", () => {
            // REF 是周三，星期一已过 → +5 天 = 周一
            const r = extractDateTime("星期一提交", REF)!;
            expect(r.dueDate.getDate()).toBe(9);
            expect(r.cleanText).toBe("提交");
        });

        it("应解析 '下周三'", () => {
            // REF 是周三，下周三 = +7 天
            const r = extractDateTime("下周三开会", REF)!;
            expect(r.dueDate.getDate()).toBe(11);
            expect(r.cleanText).toBe("开会");
        });

        it("应解析 '下周一'", () => {
            // REF 是周三，下周一 = +5+7=... 让我算一下
            // 周三=3, 目标周一=1, (1-3+7)%7=5, +7=12 天后 → 2月16日
            const r = extractDateTime("下周一汇报", REF)!;
            expect(r.dueDate.getDate()).toBe(16);
            expect(r.cleanText).toBe("汇报");
        });
    });

    describe("具体日期", () => {
        it("应解析 '3月15号'", () => {
            const r = extractDateTime("3月15号开会", REF)!;
            expect(r.dueDate.getMonth()).toBe(2); // March
            expect(r.dueDate.getDate()).toBe(15);
            expect(r.cleanText).toBe("开会");
        });

        it("应解析已过去的日期为明年", () => {
            const r = extractDateTime("1月1日庆祝", REF)!;
            expect(r.dueDate.getFullYear()).toBe(2027);
            expect(r.dueDate.getMonth()).toBe(0);
            expect(r.dueDate.getDate()).toBe(1);
        });
    });

    describe("时间", () => {
        it("应解析 '下午3点'", () => {
            const r = extractDateTime("下午3点开会", REF)!;
            expect(r.dueDate.getHours()).toBe(15);
            expect(r.dueDate.getMinutes()).toBe(0);
            expect(r.hasTime).toBe(true);
            expect(r.cleanText).toBe("开会");
        });

        it("应解析 '上午10点半'", () => {
            const r = extractDateTime("上午10点半出发", REF)!;
            expect(r.dueDate.getHours()).toBe(10);
            expect(r.dueDate.getMinutes()).toBe(30);
            expect(r.cleanText).toBe("出发");
        });

        it("应解析 '晚上8点15分'", () => {
            const r = extractDateTime("晚上8点15分吃饭", REF)!;
            expect(r.dueDate.getHours()).toBe(20);
            expect(r.dueDate.getMinutes()).toBe(15);
        });

        it("应解析不带时段的 '3点'", () => {
            const r = extractDateTime("3点开会", REF)!;
            expect(r.dueDate.getHours()).toBe(3);
            expect(r.hasTime).toBe(true);
        });
    });

    describe("日期 + 时间组合", () => {
        it("应解析 '明天下午3点'", () => {
            const r = extractDateTime("明天下午3点开会", REF)!;
            expect(r.dueDate.getDate()).toBe(5);
            expect(r.dueDate.getHours()).toBe(15);
            expect(r.hasTime).toBe(true);
            expect(r.cleanText).toBe("开会");
        });

        it("应解析 '下周五上午10点'", () => {
            // REF 周三，下周五 = +9 天 → 2月13日
            const r = extractDateTime("下周五上午10点汇报", REF)!;
            expect(r.dueDate.getDate()).toBe(13);
            expect(r.dueDate.getHours()).toBe(10);
            expect(r.cleanText).toBe("汇报");
        });
    });

    describe("边界情况", () => {
        it("无日期时间信息应返回 undefined", () => {
            expect(extractDateTime("买牛奶", REF)).toBeUndefined();
        });

        it("应清理残留的 '前'", () => {
            const r = extractDateTime("周五前交报告", REF)!;
            expect(r.cleanText).toBe("交报告");
        });
    });
});

describe("formatDueDate", () => {
    it("应格式化仅日期", () => {
        const d = new Date(2026, 1, 5);
        expect(formatDueDate(d, false)).toBe("2026-02-05");
    });

    it("应格式化日期和时间", () => {
        const d = new Date(2026, 1, 5, 15, 30);
        expect(formatDueDate(d, true)).toBe("2026-02-05 15:30");
    });
});

describe("format2DoDate", () => {
    it("应格式化仅日期为 M-D-YYYY", () => {
        const d = new Date(2026, 1, 5);
        expect(format2DoDate(d, false)).toBe("2-5-2026");
    });

    it("应格式化下午整点为 M-D-YYYY Hpm", () => {
        const d = new Date(2026, 1, 5, 15, 0);
        expect(format2DoDate(d, true)).toBe("2-5-2026 3pm");
    });

    it("应格式化上午整点为 M-D-YYYY Ham", () => {
        const d = new Date(2026, 1, 5, 10, 0);
        expect(format2DoDate(d, true)).toBe("2-5-2026 10am");
    });

    it("应格式化带分钟的时间为 M-D-YYYY H:MMam/pm", () => {
        const d = new Date(2026, 1, 5, 15, 30);
        expect(format2DoDate(d, true)).toBe("2-5-2026 3:30pm");
    });

    it("应将 12 点格式化为 12pm", () => {
        const d = new Date(2026, 1, 5, 12, 0);
        expect(format2DoDate(d, true)).toBe("2-5-2026 12pm");
    });

    it("应将 0 点格式化为 12am", () => {
        const d = new Date(2026, 1, 5, 0, 0);
        expect(format2DoDate(d, true)).toBe("2-5-2026 12am");
    });

    it("应格式化带前导零分钟", () => {
        const d = new Date(2026, 1, 5, 20, 5);
        expect(format2DoDate(d, true)).toBe("2-5-2026 8:05pm");
    });
});
