import { extractDateTime, format2DoDate, formatDueDate } from "./date-parser.js";
import type { ParsedTask, Priority } from "./types.js";

// === 中文前缀模式 ===

// 中文前缀（含列表）：匹配 "添加任务到X列表：" 等
const CN_PREFIX_WITH_LIST = /^(?:添加任务|创建待办|提醒我|记录任务|新建任务|加个任务)\s*到\s*(.+?)\s*列表\s*[：:]\s*/;

// 中文前缀：匹配 "添加任务："、"创建待办：" 等
const CN_PREFIX = /^(?:添加任务|创建待办|提醒我|记录任务|新建任务|加个任务)\s*[：:]\s*/;

// === 英文前缀模式 ===

// 英文前缀（含列表）：匹配 "add task to work list:" 等
const EN_PREFIX_WITH_LIST = /^(?:add\s+task|create\s+todo)\s+to\s+(.+?)\s+list\s*[:：]\s*/i;

// 英文前缀：匹配 "add task:"、"remind me to" 等
const EN_PREFIX = /^(?:add\s+task|create\s+todo)\s*[:：]\s*|^(?:remind\s+me\s+to|remember\s+to)\s*/i;

// === 列表提取模式 ===
const LIST_PATTERNS = [
    // 中文：列表是X / 列表为X
    /[，,]\s*列表(?:是|为)\s*(.+?)(?=[，,]|$)/,
    // 中文：到X列表：
    /到\s*(.+?)\s*列表\s*[：:]\s*/,
    // 中文：到X列表
    /到\s*(.+?)\s*列表(?=[，,]|$)/,
    // 英文：, list X
    /[,]\s*list\s+(.+?)(?=[,]|$)/i,
];

// === 标签提取模式 ===
const TAG_PATTERNS = [
    // 中文：标签是X和Y
    /[，,]\s*标签(?:是|为)\s*(.+?)(?=[，,](?!.*(?:标签|标记))|$)/,
    // 中文：标记为X
    /[，,]\s*标记(?:为|是)\s*(.+?)(?=[，,](?!.*(?:标签|标记))|$)/,
    // 英文：, tag X and Y / tags X, Y
    /[,]\s*tags?\s+(.+?)(?=[,](?!.*\btags?\b)|$)/i,
];

// 标签分隔符（支持中英文）
const TAG_SEPARATORS = /(?:\s+and\s+)|[、和,，\s]+/;

// === 优先级提取模式 ===
const PRIORITY_PATTERNS: Array<{ pattern: RegExp; priority: Priority }> = [
    { pattern: /[，,]\s*(?:很紧急|非常紧急|特别紧急|特急|加急)/, priority: "high" },
    { pattern: /[，,]\s*(?:紧急|很急)/, priority: "high" },
    { pattern: /[，,]\s*(?:很重要|非常重要|特别重要)/, priority: "high" },
    { pattern: /[，,]\s*(?:重要)/, priority: "medium" },
    { pattern: /[，,]\s*(?:不太?急|不太?紧急)/, priority: "low" },
    // 英文
    { pattern: /[,]\s*(?:urgent|high\s*priority)/i, priority: "high" },
    { pattern: /[,]\s*(?:important)/i, priority: "medium" },
    { pattern: /[,]\s*(?:low\s*priority)/i, priority: "low" },
];

/** 从自然语言输入中解析任务信息 */
export function parseTask(input: string, now?: Date): ParsedTask {
    let text = input.trim();

    // 1. 提取列表（从前缀）
    let list: string | undefined;
    const cnListMatch = text.match(CN_PREFIX_WITH_LIST);
    const enListMatch = !cnListMatch ? text.match(EN_PREFIX_WITH_LIST) : null;
    const prefixListMatch = cnListMatch || enListMatch;

    if (prefixListMatch) {
        list = prefixListMatch[1].trim();
        text = text.replace(prefixListMatch[0], "").trim();
    } else {
        // 移除普通前缀（先中文后英文）
        const cnReplaced = text.replace(CN_PREFIX, "");
        text = cnReplaced !== text ? cnReplaced : text.replace(EN_PREFIX, "");
    }

    // 2. 从正文中提取列表（仅在前缀中未提取到时）
    if (!list) {
        for (const pattern of LIST_PATTERNS) {
            const match = text.match(pattern);
            if (match) {
                list = match[1].trim();
                text = text.replace(match[0], "").trim();
                break;
            }
        }
    }

    // 3. 提取标签
    let tags: string[] | undefined;
    for (const pattern of TAG_PATTERNS) {
        const match = text.match(pattern);
        if (match) {
            tags = match[1]
                .split(TAG_SEPARATORS)
                .map((t) => t.trim())
                .filter(Boolean);
            text = text.replace(match[0], "").trim();
            break;
        }
    }

    // 4. 提取优先级
    let priority: Priority | undefined;
    for (const { pattern, priority: p } of PRIORITY_PATTERNS) {
        const match = text.match(pattern);
        if (match) {
            priority = p;
            text = text.replace(match[0], "").trim();
            break;
        }
    }

    // 5. 提取日期时间
    let dueDate: Date | undefined;
    const dtResult = extractDateTime(text, now);
    if (dtResult) {
        dueDate = dtResult.dueDate;
        text = dtResult.cleanText;
    }

    // 6. 清理标题末尾的标点
    const title = text.replace(/[，,]+$/, "").trim();

    if (!title) {
        throw new Error("无法解析任务标题，请提供有效的任务描述");
    }

    return {
        title,
        ...(list && { list }),
        ...(tags && tags.length > 0 && { tags }),
        ...(dueDate && { dueDate }),
        ...(priority && { priority }),
    };
}

/** 构造 2Do 邮件主题 */
export function buildEmailSubject(task: ParsedTask, titlePrefix?: string): string {
    let subject = task.title;

    // 日期时间 → 2Do 的 start()/due() 格式
    if (task.dueDate) {
        const hasTime = task.dueDate.getHours() !== 0 || task.dueDate.getMinutes() !== 0;
        const dateStr = format2DoDate(task.dueDate, hasTime);
        if (hasTime) {
            subject += ` start(${dateStr})`;
        }
        subject += ` due(${dateStr})`;
    }

    if (task.list) {
        subject += ` list(${task.list})`;
    }

    if (task.tags && task.tags.length > 0) {
        subject += ` tag(${task.tags.join(", ")})`;
    }

    // 如果配置了标题前缀，添加到开头
    if (titlePrefix) {
        subject = `${titlePrefix}${subject}`;
    }

    return subject;
}

/** 构造邮件正文 */
export function buildEmailBody(task: ParsedTask, rawInput?: string): string {
    const lines: string[] = [];

    if (rawInput) {
        lines.push(`任务详情：${rawInput}`);
        lines.push("");
    }

    if (task.dueDate) {
        const hasTime = task.dueDate.getHours() !== 0 || task.dueDate.getMinutes() !== 0;
        lines.push(`截止时间：${formatDueDate(task.dueDate, hasTime)}`);
    }

    if (task.priority) {
        const labels: Record<Priority, string> = { high: "高", medium: "中", low: "低" };
        lines.push(`优先级：${labels[task.priority]}`);
    }

    lines.push(`创建时间：${new Date().toISOString()}`);
    lines.push("来源：Moltbot 2Do Task Email Skill");

    return lines.join("\n");
}
