import { parseArgs } from "node:util";
import { loadConfig } from "./config.js";
import { sendTaskEmail } from "./email-sender.js";
import { parseTask } from "./task-parser.js";
import type { ParsedTask } from "./types.js";

function parseCliArgs() {
    const { values } = parseArgs({
        options: {
            raw: { type: "string", short: "r" },
            title: { type: "string", short: "t" },
            list: { type: "string", short: "l" },
            tags: { type: "string" },
        },
        strict: true,
    });
    return values;
}

async function main() {
    const args = parseCliArgs();

    let task: ParsedTask;
    let rawInput: string | undefined;

    if (args.raw) {
        // 自然语言模式：自动解析
        rawInput = args.raw;
        task = parseTask(args.raw);
    } else if (args.title) {
        // 结构化参数模式
        task = {
            title: args.title,
            ...(args.list && { list: args.list }),
            ...(args.tags && { tags: args.tags.split(",").map((t) => t.trim()) }),
        };
    } else {
        console.error("错误: 需要提供 --raw 或 --title 参数");
        process.exit(1);
    }

    const config = loadConfig();
    const result = await sendTaskEmail(config, task, rawInput);

    if (result.success) {
        console.log(`✅ 任务已发送到 2Do: ${task.title}`);
    } else {
        console.error(`❌ ${result.error}`);
        process.exit(1);
    }
}

main();
