import nodemailer from "nodemailer";
import type { AppConfig, ParsedTask, SendResult } from "./types.js";
import { buildEmailBody, buildEmailSubject } from "./task-parser.js";

/** 创建邮件传输实例 */
function createTransport(config: AppConfig) {
    const isSSL = config.smtp.port === 465;

    return nodemailer.createTransport({
        host: config.smtp.host,
        port: config.smtp.port,
        secure: isSSL,
        auth: {
            user: config.smtp.user,
            pass: config.smtp.pass,
        },
        tls: {
            rejectUnauthorized: true,
            minVersion: "TLSv1.2",
        },
    });
}

/** 发送任务邮件到 2Do */
export async function sendTaskEmail(
    config: AppConfig,
    task: ParsedTask,
    rawInput?: string,
): Promise<SendResult> {
    const transporter = createTransport(config);

    const subject = buildEmailSubject(task, config.titlePrefix);
    const body = buildEmailBody(task, rawInput);

    try {
        const info = await transporter.sendMail({
            from: config.smtp.user,
            to: config.twodoEmail,
            subject,
            text: body,
        });

        return {
            success: true,
            messageId: info.messageId,
        };
    } catch (err) {
        const message = err instanceof Error ? err.message : String(err);
        return {
            success: false,
            error: `邮件发送失败: ${message}`,
        };
    } finally {
        transporter.close();
    }
}
