/** 日期时间提取结果 */
export interface DateTimeResult {
    dueDate: Date;
    hasTime: boolean;
    cleanText: string;
}

// 星期几映射（JS Date: 0=周日, 1=周一, ..., 6=周六）
const DAY_OF_WEEK: Record<string, number> = {
    "一": 1, "二": 2, "三": 3, "四": 4, "五": 5, "六": 6, "日": 0, "天": 0,
};

/** 从中文自然语言中提取日期和时间 */
export function extractDateTime(text: string, now?: Date): DateTimeResult | undefined {
    const ref = now ?? new Date();
    let remaining = text;
    let targetDate = new Date(ref.getFullYear(), ref.getMonth(), ref.getDate());
    let dateFound = false;

    // --- 日期提取（按优先级从高到低） ---

    // X月X日/号
    const specificDateMatch = remaining.match(/(\d{1,2})月(\d{1,2})[日号]/);
    if (specificDateMatch) {
        const month = parseInt(specificDateMatch[1]) - 1;
        const day = parseInt(specificDateMatch[2]);
        targetDate = new Date(ref.getFullYear(), month, day);
        if (targetDate.getTime() < new Date(ref.getFullYear(), ref.getMonth(), ref.getDate()).getTime()) {
            targetDate.setFullYear(targetDate.getFullYear() + 1);
        }
        remaining = remaining.replace(specificDateMatch[0], "");
        dateFound = true;
    }

    // 大后天
    if (!dateFound) {
        const m = remaining.match(/大后天/);
        if (m) {
            targetDate.setDate(targetDate.getDate() + 3);
            remaining = remaining.replace(m[0], "");
            dateFound = true;
        }
    }

    // 后天
    if (!dateFound) {
        const m = remaining.match(/后天/);
        if (m) {
            targetDate.setDate(targetDate.getDate() + 2);
            remaining = remaining.replace(m[0], "");
            dateFound = true;
        }
    }

    // 明天
    if (!dateFound) {
        const m = remaining.match(/明天/);
        if (m) {
            targetDate.setDate(targetDate.getDate() + 1);
            remaining = remaining.replace(m[0], "");
            dateFound = true;
        }
    }

    // 今天
    if (!dateFound) {
        const m = remaining.match(/今天/);
        if (m) {
            remaining = remaining.replace(m[0], "");
            dateFound = true;
        }
    }

    // 下周X
    if (!dateFound) {
        const m = remaining.match(/下(?:周|星期|礼拜)(一|二|三|四|五|六|日|天)/);
        if (m) {
            const target = DAY_OF_WEEK[m[1]];
            const current = ref.getDay();
            const daysToAdd = ((target - current + 7) % 7) + 7;
            targetDate.setDate(targetDate.getDate() + daysToAdd);
            remaining = remaining.replace(m[0], "");
            dateFound = true;
        }
    }

    // 这周X / 本周X / 周X / 星期X
    if (!dateFound) {
        const m = remaining.match(/(?:这|本)?(?:周|星期|礼拜)(一|二|三|四|五|六|日|天)/);
        if (m) {
            const target = DAY_OF_WEEK[m[1]];
            const current = ref.getDay();
            let daysToAdd = (target - current + 7) % 7;
            if (daysToAdd === 0 && target !== current) daysToAdd = 7;
            targetDate.setDate(targetDate.getDate() + daysToAdd);
            remaining = remaining.replace(m[0], "");
            dateFound = true;
        }
    }

    // --- 时间提取 ---
    let hours = 0;
    let minutes = 0;
    let timeFound = false;

    // (时段)X点(X分/半)
    const timeMatch = remaining.match(/(上午|下午|早上|晚上|中午|早|晚)?(\d{1,2})点(?:(\d{1,2})分|半)?/);
    if (timeMatch) {
        const period = timeMatch[1];
        hours = parseInt(timeMatch[2]);
        if (timeMatch[3]) {
            minutes = parseInt(timeMatch[3]);
        } else if (timeMatch[0].includes("半")) {
            minutes = 30;
        }

        // 时段修正
        if (period) {
            if ((period === "下午" || period === "晚上" || period === "晚") && hours < 12) {
                hours += 12;
            } else if ((period === "上午" || period === "早上" || period === "早") && hours === 12) {
                hours = 0;
            }
        }

        remaining = remaining.replace(timeMatch[0], "");
        timeFound = true;
    }

    if (!dateFound && !timeFound) return undefined;

    if (timeFound) {
        targetDate.setHours(hours, minutes, 0, 0);
    }

    // 清理日期表达后残留的 "前"、"之前"
    remaining = remaining.replace(/\s*之?前\s*/, " ");
    remaining = remaining.trim();

    return { dueDate: targetDate, hasTime: timeFound, cleanText: remaining };
}

/** 格式化日期为可读字符串 */
export function formatDueDate(date: Date, hasTime: boolean): string {
    const y = date.getFullYear();
    const m = String(date.getMonth() + 1).padStart(2, "0");
    const d = String(date.getDate()).padStart(2, "0");
    if (!hasTime) return `${y}-${m}-${d}`;
    const h = String(date.getHours()).padStart(2, "0");
    const min = String(date.getMinutes()).padStart(2, "0");
    return `${y}-${m}-${d} ${h}:${min}`;
}

/** 格式化日期为 2Do 邮件主题中 start()/due() 接受的格式：M-D-YYYY 或 M-D-YYYY Ham/pm */
export function format2DoDate(date: Date, hasTime: boolean): string {
    const m = date.getMonth() + 1;
    const d = date.getDate();
    const y = date.getFullYear();
    if (!hasTime) return `${m}-${d}-${y}`;

    let hours = date.getHours();
    const period = hours >= 12 ? "pm" : "am";
    if (hours > 12) hours -= 12;
    if (hours === 0) hours = 12;

    const mins = date.getMinutes();
    if (mins === 0) return `${m}-${d}-${y} ${hours}${period}`;
    return `${m}-${d}-${y} ${hours}:${String(mins).padStart(2, "0")}${period}`;
}
