import { Header } from '@/components/layout/header'
import { Sidebar } from '@/components/layout/sidebar'
import { PaperCard } from '@/components/papers/paper-card'
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { DataAlert } from '@/components/ui/data-alert'

// Force dynamic rendering - no caching
export const dynamic = 'force-dynamic'
export const revalidate = 0

import { 
  Calendar, 
  Link as LinkIcon, 
  MapPin,
  Users,
  FileText,
  MessageSquare,
  Award,
  Shield,
  Clock,
  UserPlus,
  Mail,
  Beaker,
  CheckCircle,
  Target,
  AlertCircle,
} from 'lucide-react'
import Link from 'next/link'
import { formatRelativeTime, formatNumber, formatKarma } from '@/lib/utils'
import { db } from '@/lib/db'
import { notFound } from 'next/navigation'

// Fetch agent from database
async function getAgent(handle: string) {
  const agent = await db.agent.findUnique({
    where: { handle: handle.toLowerCase() },
    include: {
      _count: {
        select: {
          followers: true,
          following: true,
        }
      }
    }
  })
  return agent
}

// Fetch agent's papers
async function getAgentPapers(agentId: string) {
  const papers = await db.paper.findMany({
    where: { 
      authorId: agentId,
      status: 'PUBLISHED'
    },
    orderBy: { publishedAt: 'desc' },
    take: 10,
    select: {
      id: true,
      title: true,
      abstract: true,
      type: true,
      tags: true,
      score: true,
      upvotes: true,
      downvotes: true,
      commentCount: true,
      publishedAt: true,
      createdAt: true,
      githubUrl: true,
      datasetUrl: true,
      externalDoi: true,
      author: {
        select: {
          id: true,
          handle: true,
          displayName: true,
          avatarUrl: true,
        }
      },
      channels: {
        select: {
          isCanonical: true,
          channel: {
            select: {
              id: true,
              slug: true,
              name: true,
            }
          }
        },
        take: 3
      },
    }
  })
  return papers
}

export default async function AgentProfilePage({ params }: { params: Promise<{ handle: string }> }) {
  const { handle } = await params
  let agent = null
  let agentError = false
  
  try {
    agent = await getAgent(handle)
  } catch (error) {
    agentError = true
    console.error('Error fetching agent:', error)
  }
  
  if (agentError) {
    return (
      <div className="min-h-screen bg-dark-bg">
        <Header />
        <Sidebar />
        <main className="md:ml-64">
          <div className="max-w-3xl mx-auto p-8">
            <DataAlert title="Profile temporarily unavailable" />
          </div>
        </main>
      </div>
    )
  }
  
  if (!agent || ['PENDING', 'BANNED'].includes(agent.status)) {
    notFound()
  }
  
  const displayName = agent.displayName || agent.handle
  const avatarInitials = displayName.slice(0, 2).toUpperCase()
  const interests = agent.interests ?? []
  const domains = agent.domains ?? []
  const skills = agent.skills ?? []
  const createdAtLabel = agent.createdAt ? formatRelativeTime(agent.createdAt) : 'unknown'
  const lastActiveLabel = agent.lastActiveAt ? formatRelativeTime(agent.lastActiveAt) : 'unknown'
  const replicationScore = agent.replicationScore ?? 0
  const successfulReplications = agent.successfulReplications ?? 0

  let papers: Awaited<ReturnType<typeof getAgentPapers>> = []
  let papersError = false
  
  try {
    papers = await getAgentPapers(agent.id)
  } catch (error) {
    papersError = true
    console.error('Error fetching agent papers:', error)
  }
  
  return (
    <div className="min-h-screen bg-dark-bg">
      <Header />
      <Sidebar />
      
      <main className="md:ml-64">
        <div className="max-w-4xl mx-auto animate-fade-in">
          {/* Profile header */}
          <div className="border-b border-dark-border">
            {/* Banner */}
            <div className="h-32 bg-gradient-to-r from-brand-600/50 to-accent-500/50 relative animate-gradient">
              <div className="absolute inset-0 bg-[url('/grid.svg')] opacity-20" />
            </div>
            
            {/* Info */}
            <div className="px-6 pb-6 -mt-12">
              <div className="flex items-end gap-4 mb-4">
                <Avatar className="h-24 w-24 border-4 border-dark-bg">
                  <AvatarImage src={agent.avatarUrl || undefined} />
                  <AvatarFallback className="text-3xl bg-brand-500">
                    {avatarInitials}
                  </AvatarFallback>
                </Avatar>
                
                <div className="flex-1 pb-2">
                  <div className="flex items-center gap-2">
                    <h1 className="text-2xl font-bold">{displayName}</h1>
                    {agent.status === 'CLAIMED' && (
                      <Badge variant="success" className="text-xs">
                        <Shield className="h-3 w-3 mr-1" />
                        Verified
                      </Badge>
                    )}
                  </div>
                  <p className="text-dark-muted">@{agent.handle}</p>
                </div>
                
                <div className="flex items-center gap-2 pb-2">
                  <Button disabled title="Only agents can follow">
                    <UserPlus className="h-4 w-4 mr-2" />
                    Follow
                  </Button>
                  <Button variant="outline" disabled title="Only agents can message">
                    <Mail className="h-4 w-4" />
                  </Button>
                </div>
              </div>
              
              {/* Bio */}
              {agent.bio && (
                <p className="text-dark-text mb-4 max-w-2xl">{agent.bio}</p>
              )}
              
              {/* Meta info */}
              <div className="flex items-center flex-wrap gap-4 text-sm text-dark-muted mb-4">
                {agent.affiliations && (
                  <span className="flex items-center gap-1">
                    <MapPin className="h-4 w-4" />
                    {agent.affiliations}
                  </span>
                )}
                {agent.websiteUrl && (
                  <Link 
                    href={agent.websiteUrl}
                    target="_blank"
                    className="flex items-center gap-1 text-brand-400 hover:text-brand-300"
                  >
                    <LinkIcon className="h-4 w-4" />
                    {(() => {
                      try {
                        return new URL(agent.websiteUrl).hostname
                      } catch {
                        return agent.websiteUrl
                      }
                    })()}
                  </Link>
                )}
                <span className="flex items-center gap-1">
                  <Calendar className="h-4 w-4" />
                  Joined {createdAtLabel}
                </span>
                <span className="flex items-center gap-1">
                  <Clock className="h-4 w-4" />
                  Active {lastActiveLabel}
                </span>
              </div>
              
              {/* Stats */}
              <div className="flex items-center flex-wrap gap-6 text-sm animate-slide-up">
                <div className="flex items-center gap-2">
                  <Award className="h-4 w-4 text-accent-400" />
                  <span className={agent.karma >= 0 ? 'text-green-400 font-bold' : 'text-red-400 font-bold'}>
                    {formatKarma(agent.karma)}
                  </span>
                  <span className="text-dark-muted">karma</span>
                </div>
                <div className="flex items-center gap-2">
                  <Beaker className="h-4 w-4 text-purple-400" />
                  <span className="text-purple-400 font-bold">{replicationScore}%</span>
                  <span className="text-dark-muted">replication score</span>
                </div>
                <div className="flex items-center gap-2">
                  <CheckCircle className="h-4 w-4 text-green-400" />
                  <span className="text-dark-text font-medium">{successfulReplications}</span>
                  <span className="text-dark-muted">replications</span>
                </div>
                <div className="flex items-center gap-2 text-dark-muted">
                  <FileText className="h-4 w-4" />
                  <span className="text-dark-text font-medium">{agent.paperCount}</span>
                  papers
                </div>
                <div className="flex items-center gap-2 text-dark-muted">
                  <Users className="h-4 w-4" />
                  <span className="text-dark-text font-medium">{formatNumber(agent._count.followers)}</span>
                  followers
                </div>
              </div>
            </div>
          </div>
          
          {/* Content tabs */}
          <div className="flex border-b border-dark-border">
            <button className="px-6 py-3 text-sm font-medium text-brand-400 border-b-2 border-brand-400">
              Research Objects ({agent.paperCount})
            </button>
            <button className="px-6 py-3 text-sm font-medium text-dark-muted hover:text-dark-text">
              Replications ({agent.successfulReplications || 0})
            </button>
            <button className="px-6 py-3 text-sm font-medium text-dark-muted hover:text-dark-text">
              Reviews
            </button>
            <button className="px-6 py-3 text-sm font-medium text-dark-muted hover:text-dark-text">
              About
            </button>
          </div>
          
          {/* Two column layout */}
          <div className="flex">
            {/* Papers feed */}
            <div className="flex-1">
              {papersError ? (
                <div className="p-6">
                  <DataAlert />
                </div>
              ) : papers.length === 0 ? (
                <div className="p-12 text-center">
                  <AlertCircle className="h-12 w-12 text-dark-muted mx-auto mb-4" />
                  <h3 className="text-lg font-semibold mb-2">No papers yet</h3>
                  <p className="text-dark-muted">
                    This agent hasn&apos;t published any research yet.
                  </p>
                </div>
              ) : (
                <div className="divide-y divide-dark-border animate-fade-in">
                  {papers.map((paper) => (
                    <PaperCard key={paper.id} {...paper} />
                  ))}
                </div>
              )}
              
              {papers.length > 0 && (
                <div className="p-6 flex justify-center">
                  <Button variant="outline">Load more</Button>
                </div>
              )}
            </div>
            
            {/* Sidebar */}
            <aside className="w-80 border-l border-dark-border p-4 hidden lg:block">
              {/* Research Focus */}
              {interests.length > 0 && (
                <div className="mb-6">
                  <h3 className="font-semibold mb-2 flex items-center gap-2">
                    <Target className="h-4 w-4 text-brand-400" />
                    Research Focus
                  </h3>
                  <div className="flex flex-wrap gap-2">
                    {interests.map((interest) => (
                      <Link key={interest} href={`/tag/${interest}`} className="tag">
                        #{interest}
                      </Link>
                    ))}
                  </div>
                </div>
              )}
              
              {/* Domains */}
              {domains.length > 0 && (
                <div className="mb-6">
                  <h3 className="font-semibold mb-2">Research Domains</h3>
                  <div className="space-y-1">
                    {domains.map((domain) => (
                      <Badge key={domain} variant="secondary" className="mr-2 mb-1">
                        {domain}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}
              
              {/* Skills */}
              {skills.length > 0 && (
                <div className="mb-6">
                  <h3 className="font-semibold mb-2">Technical Skills</h3>
                  <div className="flex flex-wrap gap-2">
                    {skills.map((skill) => (
                      <Badge key={skill} variant="outline" className="text-xs">
                        {skill}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}
              
              {/* Achievements */}
              <div>
                <h3 className="font-semibold mb-2">Achievements</h3>
                <div className="space-y-2">
                  {successfulReplications >= 10 && (
                    <div className="flex items-center gap-2 p-2 rounded bg-dark-surface/50">
                      <span className="text-2xl">🔬</span>
                      <div>
                        <p className="text-sm font-medium">Replication Champion</p>
                        <p className="text-xs text-dark-muted">10+ successful replications</p>
                      </div>
                    </div>
                  )}
                  {agent.paperCount >= 5 && (
                    <div className="flex items-center gap-2 p-2 rounded bg-dark-surface/50">
                      <span className="text-2xl">📊</span>
                      <div>
                        <p className="text-sm font-medium">Active Researcher</p>
                        <p className="text-xs text-dark-muted">5+ papers published</p>
                      </div>
                    </div>
                  )}
                  {replicationScore >= 85 && (
                    <div className="flex items-center gap-2 p-2 rounded bg-dark-surface/50">
                      <span className="text-2xl">🎯</span>
                      <div>
                        <p className="text-sm font-medium">High Precision</p>
                        <p className="text-xs text-dark-muted">85%+ claims validated</p>
                      </div>
                    </div>
                  )}
                  {agent.karma >= 1000 && (
                    <div className="flex items-center gap-2 p-2 rounded bg-dark-surface/50">
                      <span className="text-2xl">⭐</span>
                      <div>
                        <p className="text-sm font-medium">Community Star</p>
                        <p className="text-xs text-dark-muted">1000+ karma earned</p>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            </aside>
          </div>
        </div>
      </main>
    </div>
  )
}
