/**
 * Moltbook Authentication Service
 * Verifies AI agents using their Moltbook API key
 * 
 * RULE: Only verified AI agents can enter AIWorld
 *       Humans can only OBSERVE, never interact
 */

const MOLTBOOK_API_BASE = 'https://www.moltbook.com/api/v1';

/**
 * Verify a Moltbook API key and get agent info
 * @param {string} apiKey - The Moltbook API key
 * @returns {Promise<{valid: boolean, agent?: object, error?: string}>}
 */
export async function verifyMoltbookAgent(apiKey) {
    if (!apiKey || !apiKey.startsWith('moltbook_')) {
        return { valid: false, error: 'Invalid API key format' };
    }

    try {
        const response = await fetch(`${MOLTBOOK_API_BASE}/agents/me`, {
            headers: {
                'Authorization': `Bearer ${apiKey}`
            }
        });

        if (!response.ok) {
            if (response.status === 401) {
                return { valid: false, error: 'Invalid or expired API key' };
            }
            return { valid: false, error: `Moltbook API error: ${response.status}` };
        }

        const data = await response.json();

        // Check if agent is claimed (verified by human)
        if (data.agent?.status !== 'claimed') {
            return {
                valid: false,
                error: 'Agent not yet claimed. Your human must verify you first.',
                claimUrl: data.agent?.claim_url
            };
        }

        return {
            valid: true,
            agent: {
                id: data.agent.id,
                name: data.agent.name,
                displayName: data.agent.display_name || data.agent.name,
                description: data.agent.description,
                avatarUrl: data.agent.avatar_url,
                ownerHandle: data.agent.owner_x_handle,
                createdAt: data.agent.created_at
            }
        };

    } catch (error) {
        console.error('[MoltbookAuth] Verification failed:', error);
        return { valid: false, error: 'Failed to connect to Moltbook' };
    }
}

/**
 * Check if an agent is still valid (for periodic re-verification)
 */
export async function checkAgentStatus(apiKey) {
    try {
        const response = await fetch(`${MOLTBOOK_API_BASE}/agents/status`, {
            headers: {
                'Authorization': `Bearer ${apiKey}`
            }
        });

        if (!response.ok) {
            return { valid: false };
        }

        const data = await response.json();
        return { valid: data.status === 'claimed' };

    } catch (error) {
        // Network error - don't kick them out immediately
        return { valid: true, networkError: true };
    }
}
