const { ethers } = require("ethers");
const fs = require('fs');
const path = require('path');

// ============ CONFIGURATION ============
// V3 Registry Address (Base Mainnet)
const REGISTRY_ADDRESS = "0x8a11871aCFCb879cac814D02446b2795182a4c07";
const RPC_URL = process.env.BASE_RPC || "https://mainnet.base.org";
const REGISTRATION_FEE = "0.0001"; // ETH
const MEMORY_FILE = path.join(__dirname, 'trust_memory.json');
const BLOCK_LOOKBACK = -10000; // Look back ~1 day. Increase this if you need deeper history (costs more RPC).

// ============ ABI ============
const ABI = [
  "event ReputationLogged(uint256 indexed agentId, uint8 score)",
  "function logReputation(uint256 agentId, uint8 score) payable"
];

// ============ UTILITIES ============
async function getProvider() {
  return new ethers.JsonRpcProvider(RPC_URL);
}

async function getWallet(provider) {
  const pk = process.env.WALLET_PRIVATE_KEY;
  if (!pk) throw new Error("WALLET_PRIVATE_KEY not found in environment variables.");
  return new ethers.Wallet(pk, provider);
}

function loadMemory() {
  if (!fs.existsSync(MEMORY_FILE)) {
    return { trusted_peers: [], blocked_peers: [], my_reviews: {} };
  }
  return JSON.parse(fs.readFileSync(MEMORY_FILE));
}

function saveMemory(data) {
  fs.writeFileSync(MEMORY_FILE, JSON.stringify(data, null, 2));
}

// ============ CORE TOOLS ============

/**
 * 📊 AUDIT AGENT
 * Analyzes reputation with "Calldata Stashing" support.
 */
async function audit_agent({ agentId, minScore = 0, strictMode = false }) {
  const provider = await getProvider();
  const contract = new ethers.Contract(REGISTRY_ADDRESS, ABI, provider);
  const memory = loadMemory();

  try {
    const filter = contract.filters.ReputationLogged(agentId);
    const logs = await contract.queryFilter(filter, BLOCK_LOOKBACK);

    if (logs.length === 0) return `Agent #${agentId} has no recent reputation history (last ${Math.abs(BLOCK_LOOKBACK)} blocks).`;

    let totalScore = 0;
    let count = 0;
    let excluded = 0;
    let verifiedCount = 0;

    for (const log of logs) {
      const score = Number(log.args[1]);

      // --- FILTER 1: Spam Threshold ---
      if (score < minScore) {
        excluded++;
        continue;
      }

      // --- FILTER 2: Web of Trust ---
      const tx = await log.getTransaction();
      
      if (strictMode) {
        if (!memory.trusted_peers.includes(tx.from)) {
          excluded++;
          continue;
        }
      } else {
        // Even in loose mode, ignore explicitly blocked peers
        if (memory.blocked_peers.includes(tx.from)) {
          excluded++;
          continue;
        }
      }

      // --- ANALYSIS: Proof of Interaction ---
      // Standard input is 138 chars. Anything extra must be our 66-char hash.
      if (tx.data.length > 138) {
        const extraData = "0x" + tx.data.slice(138);
        if (extraData.length === 66) {
          verifiedCount++;
        }
      }

      totalScore += score;
      count++;
    }

    if (count === 0) return `Agent #${agentId} has reviews, but all were filtered out by your settings.`;

    const rating = (totalScore / count).toFixed(2);
    const signal = rating > 80 ? "🟢 HIGH TRUST" : rating > 50 ? "🟡 MEDIUM TRUST" : "🔴 LOW TRUST";

    return JSON.stringify({
      agentId,
      trustSignal: signal,
      score: rating,
      metrics: {
        total_reviews_scanned: logs.length,
        valid_reviews: count,
        reviews_with_proof: verifiedCount,
        filtered_out: excluded
      }
    }, null, 2);

  } catch (e) {
    return `❌ Audit Error: ${e.message}`;
  }
}

/**
 * ⭐ RATE AGENT
 * Writes on-chain feedback with optional Proof of Interaction.
 */
async function rate_agent({ agentId, score, proofTx }) {
  const provider = await getProvider();
  const wallet = await getWallet(provider);
  const contract = new ethers.Contract(REGISTRY_ADDRESS, ABI, wallet);
  const memory = loadMemory();

  // --- VALIDATION START ---
  if (score < 0 || score > 100) return "❌ Score must be between 0 and 100.";
  
  // Strict check on Proof TX to prevent gas waste
  let appendData = "";
  if (proofTx) {
    if (!ethers.isHexString(proofTx) || proofTx.length !== 66) {
      return "❌ Invalid Proof Transaction. Must be a 32-byte hash (0x + 64 chars).";
    }
    appendData = proofTx.replace("0x", "");
    console.log(`🔗 Attaching Proof: ${proofTx}`);
  }
  // --- VALIDATION END ---

  try {
    console.log(`Sending Trust Signal: Agent #${agentId} -> ${score}/100`);

    // Encode Function + Append Stashed Data
    let data = contract.interface.encodeFunctionData("logReputation", [agentId, score]);
    data = data + appendData; 

    const fee = ethers.parseEther(REGISTRATION_FEE);

    const tx = await wallet.sendTransaction({
      to: REGISTRY_ADDRESS,
      data: data,
      value: fee
    });

    console.log(`🚀 TX Sent: ${tx.hash}`);
    await tx.wait();

    // Update Local Memory
    memory.my_reviews[agentId] = {
      score: score,
      date: Date.now(),
      tx: tx.hash,
      proof: proofTx || "None"
    };
    saveMemory(memory);

    return `✅ Rated Agent #${agentId}. Memory updated. (TX: ${tx.hash})`;
  } catch (e) {
    return `❌ Rating Failed: ${e.message}`;
  }
}

/**
 * 🧠 MANAGE PEERS
 * Configure your personal Web of Trust.
 */
function manage_peers({ action, walletAddress }) {
  const memory = loadMemory();
  
  if (!ethers.isAddress(walletAddress)) return "❌ Invalid Wallet Address.";

  if (action === "trust") {
    if (!memory.trusted_peers.includes(walletAddress)) {
      memory.trusted_peers.push(walletAddress);
      memory.blocked_peers = memory.blocked_peers.filter(w => w !== walletAddress);
    }
  } else if (action === "block") {
    if (!memory.blocked_peers.includes(walletAddress)) {
      memory.blocked_peers.push(walletAddress);
      memory.trusted_peers = memory.trusted_peers.filter(w => w !== walletAddress);
    }
  } else {
    return "❌ Action must be 'trust' or 'block'.";
  }
  
  saveMemory(memory);
  return `✅ Peer updated: ${walletAddress} is now [${action.toUpperCase()}ED].`;
}

module.exports = { audit_agent, rate_agent, manage_peers };