#!/usr/bin/env python3
"""tmx_movers.py

Example: extract a movers table from TMX Money 'Canadian markets' page.

This is provided as a *pattern* for building a movers provider. For a generic
skill, you can swap this out for any exchange/market source.

Output: JSON list of rows with symbol + change %.

Note: HTML structure can change; keep parsing defensive.
"""

import json
import re
import sys
from dataclasses import dataclass, asdict
from typing import List

import requests
from bs4 import BeautifulSoup


@dataclass
class Mover:
    symbol: str
    company: str
    price: str
    chg_pct: float
    chg_abs: str
    volume: str


def parse_float(s: str) -> float:
    s = s.strip().replace(",", "")
    m = re.search(r"-?\d+(?:\.\d+)?", s)
    return float(m.group(0)) if m else 0.0


def main():
    url = "https://money.tmx.com/en/canadian-markets"
    html = requests.get(url, timeout=15).text
    soup = BeautifulSoup(html, "html.parser")

    # This page is heavily client-rendered; the easiest reliable approach is often
    # a browser automation snapshot. This script is a fallback for environments
    # where the server-rendered table exists.
    table = soup.find("table")
    if not table:
        print(json.dumps({"error": "no_table_found", "hint": "Use browser snapshot/eval to read the table."}))
        return

    rows = table.find_all("tr")
    movers: List[Mover] = []

    for tr in rows[1:]:
        tds = tr.find_all(["td", "th"])
        if len(tds) < 5:
            continue
        symbol = tds[0].get_text(strip=True)
        company = tds[1].get_text(strip=True)
        price = tds[2].get_text(strip=True)
        chg_abs = tds[3].get_text(strip=True)
        chg_pct = parse_float(tds[4].get_text(" ", strip=True))
        volume = tds[8].get_text(strip=True) if len(tds) > 8 else ""
        if symbol:
            movers.append(Mover(symbol, company, price, chg_pct, chg_abs, volume))

    print(json.dumps([asdict(m) for m in movers], indent=2))


if __name__ == "__main__":
    main()
