#!/usr/bin/env python3
"""price_tape.py

Minimal yfinance-based price tape for a list of tickers.
Outputs JSON or a markdown-friendly table.

Usage:
  python price_tape.py --tickers "^GSPC ^IXIC BTC-USD ETH-USD CL=F GC=F" --period 6mo

Notes:
- yfinance coverage varies by market.
- For Canadian tickers, you may need suffixes like .TO.
"""

import argparse
import json
from dataclasses import asdict, dataclass
from typing import List, Optional

import pandas as pd
import yfinance as yf


def rsi(series: pd.Series, period: int = 14) -> pd.Series:
    delta = series.diff()
    up = delta.clip(lower=0)
    down = -delta.clip(upper=0)
    roll_up = up.ewm(alpha=1 / period, adjust=False).mean()
    roll_down = down.ewm(alpha=1 / period, adjust=False).mean()
    rs = roll_up / roll_down
    return 100 - (100 / (1 + rs))


@dataclass
class TapeRow:
    ticker: str
    last: Optional[float]
    d1_pct: Optional[float]
    w1_pct: Optional[float]
    m1_pct: Optional[float]
    rsi14: Optional[float]
    ma20: Optional[float]
    ma50: Optional[float]
    pattern: str


def compute_row(ticker: str, period: str) -> TapeRow:
    df = yf.download(ticker, period=period, interval="1d", progress=False)
    if df is None or len(df) < 60:
        return TapeRow(ticker, None, None, None, None, None, None, None, "NA")

    # Flatten columns if MultiIndex
    if isinstance(df.columns, pd.MultiIndex):
        df.columns = [c[0] for c in df.columns]

    close = df["Close"].astype(float)
    ma20 = close.rolling(20).mean()
    ma50 = close.rolling(50).mean()
    r = rsi(close, 14)

    last = float(close.iloc[-1])
    d1 = float(close.pct_change().iloc[-1] * 100)
    w1 = float(close.pct_change(5).iloc[-1] * 100)
    m1 = float(close.pct_change(21).iloc[-1] * 100)

    ma20_last = float(ma20.iloc[-1]) if pd.notna(ma20.iloc[-1]) else None
    ma50_last = float(ma50.iloc[-1]) if pd.notna(ma50.iloc[-1]) else None
    r_last = float(r.iloc[-1]) if pd.notna(r.iloc[-1]) else None

    pattern = "WAIT"
    if ma20_last is not None and ma50_last is not None and r_last is not None:
        if last > ma20_last > ma50_last and r_last >= 50:
            pattern = "BUY"
        elif last < ma20_last < ma50_last and r_last <= 50:
            pattern = "SELL"

    return TapeRow(ticker, last, d1, w1, m1, r_last, ma20_last, ma50_last, pattern)


def to_markdown(rows: List[TapeRow]) -> str:
    headers = ["Ticker", "Pattern", "Last", "1D%", "1W%", "1M%", "RSI", "MA20", "MA50"]
    lines = ["| " + " | ".join(headers) + " |", "|" + "---|" * len(headers)]

    def fmt(x, nd=2):
        if x is None:
            return "—"
        return f"{x:.{nd}f}"

    for r in rows:
        lines.append(
            "| "
            + " | ".join(
                [
                    r.ticker,
                    r.pattern,
                    fmt(r.last),
                    fmt(r.d1_pct),
                    fmt(r.w1_pct),
                    fmt(r.m1_pct),
                    fmt(r.rsi14, 1),
                    fmt(r.ma20),
                    fmt(r.ma50),
                ]
            )
            + " |"
        )
    return "\n".join(lines)


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--tickers", required=True, help="Space-separated tickers")
    ap.add_argument("--period", default="6mo")
    ap.add_argument("--format", choices=["json", "md"], default="json")
    args = ap.parse_args()

    tickers = [t.strip() for t in args.tickers.split() if t.strip()]
    rows = [compute_row(t, args.period) for t in tickers]

    if args.format == "md":
        print(to_markdown(rows))
    else:
        print(json.dumps([asdict(r) for r in rows], indent=2))


if __name__ == "__main__":
    main()
