#!/usr/bin/env python3
"""movers_yahoo.py

Fetch top movers (gainers/losers/actives) from Yahoo Finance's public screener endpoints.

This is a *best-effort* free data source. Yahoo may rate-limit or change endpoints.

Examples:
  python movers_yahoo.py --type gainers --count 10
  python movers_yahoo.py --type losers --count 10
  python movers_yahoo.py --type actives --count 10

Output: JSON list of tickers with price + % change + volume + market cap when available.

Note:
- These are typically US-centric lists. For other exchanges, swap movers provider.
- Pair with price_tape.py (yfinance) to enrich history/RSI/MA.
"""

import argparse
import json
import sys
from typing import Any, Dict, List

import requests

SCR_IDS = {
    "gainers": "day_gainers",
    "losers": "day_losers",
    "actives": "most_actives",
}


def fetch(scr_id: str, count: int) -> List[Dict[str, Any]]:
    url = "https://query1.finance.yahoo.com/v1/finance/screener/predefined/saved"
    params = {"scrIds": scr_id, "count": count}
    r = requests.get(url, params=params, timeout=15, headers={"User-Agent": "Mozilla/5.0"})
    r.raise_for_status()
    j = r.json()
    quotes = (
        j.get("finance", {})
        .get("result", [{}])[0]
        .get("quotes", [])
    )

    out: List[Dict[str, Any]] = []
    for q in quotes:
        out.append(
            {
                "symbol": q.get("symbol"),
                "shortName": q.get("shortName") or q.get("longName"),
                "regularMarketPrice": q.get("regularMarketPrice"),
                "regularMarketChangePercent": q.get("regularMarketChangePercent"),
                "regularMarketVolume": q.get("regularMarketVolume"),
                "marketCap": q.get("marketCap"),
                "currency": q.get("currency"),
                "exchange": q.get("fullExchangeName") or q.get("exchange"),
            }
        )
    return out


def main() -> None:
    ap = argparse.ArgumentParser()
    ap.add_argument("--type", choices=sorted(SCR_IDS.keys()), required=True)
    ap.add_argument("--count", type=int, default=10)
    args = ap.parse_args()

    scr_id = SCR_IDS[args.type]
    try:
        rows = fetch(scr_id, args.count)
    except Exception as e:
        print(json.dumps({"error": str(e), "hint": "Yahoo endpoint may be rate-limited/changed. Try later or use another movers provider."}))
        sys.exit(1)

    print(json.dumps({"type": args.type, "source": "yahoo", "items": rows}, indent=2))


if __name__ == "__main__":
    main()
