# Model Benchmarks - Global AI Intelligence Hub 🧠

**Real-time AI model capability tracking for intelligent compute routing and cost optimization.**

[![OpenClaw Skill](https://img.shields.io/badge/OpenClaw-Skill-blue)](https://openclaw.ai)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![ClawHub](https://img.shields.io/badge/ClawHub-Ready-green)](https://clawhub.ai)

## 🚀 Why This Skill?

Stop overpaying for AI! This skill provides real-time intelligence to route your OpenClaw tasks to the most cost-effective models:

- **💰 Cost Savings**: Users report 60-95% reduction in AI API costs
- **📊 Data-Driven**: Real benchmark data from LMSYS Arena, BigCode, HuggingFace
- **🎯 Task-Optimized**: Smart recommendations for coding, writing, analysis, and more
- **⚡ Always Fresh**: Daily updates with latest model performance data

## ⚡ Quick Start

```bash
# Install skill (via ClawHub or manual download)
openclaw skills install model-benchmarks

# Get latest AI intelligence
python3 skills/model-benchmarks/scripts/run.py fetch

# Find best model for coding
python3 skills/model-benchmarks/scripts/run.py recommend --task coding

# Check any model's capabilities
python3 skills/model-benchmarks/scripts/run.py query --model gpt-4o
```

## 📊 Sample Results

```bash
$ python3 scripts/run.py recommend --task coding

🏆 Top 3 recommendations for coding:
1. gemini-2.0-flash
   Task Score: 81.5/100
   Cost Efficiency: 445.33
   Avg Price: $0.19/1M tokens

2. claude-3.5-sonnet  
   Task Score: 92.0/100
   Cost Efficiency: 10.28
   Avg Price: $9.00/1M tokens

3. gpt-4o
   Task Score: 82.8/100
   Cost Efficiency: 13.48
   Avg Price: $6.25/1M tokens
```

## 🔧 Integration Examples

### Auto-Configure OpenClaw
```bash
# Get optimal model and configure OpenClaw
BEST_MODEL=$(python3 scripts/run.py recommend --task coding --format json | jq -r '.recommendations[0].model')
openclaw config set agents.defaults.model.primary "openrouter/$BEST_MODEL"
```

### Daily Optimization
```bash
# Add to crontab for automatic optimization
0 8 * * * cd ~/.openclaw/workspace && python3 skills/model-benchmarks/scripts/run.py fetch
```

### Cost Monitoring
```python
import subprocess
import json

def get_optimal_model(task_type, max_cost_per_1m=5.0):
    cmd = ["python3", "scripts/run.py", "recommend", "--task", task_type, "--format", "json"]
    result = subprocess.run(cmd, capture_output=True, text=True)
    recommendations = json.loads(result.stdout)
    
    for model in recommendations["recommendations"]:
        if model["avg_price"] <= max_cost_per_1m:
            return model["model"]
    
    return recommendations["recommendations"][-1]["model"]
```

## 📈 Supported Benchmarks

| Platform | Models | Capabilities Tracked | Update Frequency |
|----------|--------|---------------------|------------------|
| LMSYS Chatbot Arena | 100+ | General, Reasoning, Creative | Daily |
| BigCode Leaderboard | 50+ | Coding (HumanEval, MBPP) | Weekly |
| Open LLM Leaderboard | 200+ | Knowledge, Comprehension | Daily |
| Alpaca Eval | 80+ | Instruction Following | Weekly |

## 🎯 Task Types

- **coding** — Programming and technical tasks
- **writing** — Creative writing and content generation  
- **analysis** — Data analysis and reasoning
- **translation** — Language translation tasks
- **math** — Mathematical problem solving
- **creative** — Creative and artistic tasks
- **simple** — Basic questions and simple tasks

## 📁 File Structure

```
model-benchmarks/
├── SKILL.md              # Skill metadata and documentation
├── README.md             # This file
├── scripts/
│   └── run.py            # Main script with all functionality
├── examples/             # Integration examples
│   ├── daily-optimization.sh
│   └── integration-examples.md
└── LICENSE               # MIT License
```

## 🛡️ Privacy & Security

- **No personal data collected** — Only public benchmark results
- **Local processing** — All analysis runs on your machine  
- **No external dependencies** — Uses Python standard library only
- **Open source** — Full transparency, audit the code yourself

## 🔮 Roadmap

- **v1.1**: Real-time pricing from OpenRouter/Anthropic APIs
- **v1.2**: Custom benchmark suite for domain-specific tasks
- **v1.3**: Multi-provider cost comparison (Direct APIs vs Proxies)
- **v2.0**: Predictive model performance based on task characteristics

## 🤝 Contributing

1. Fork this skill
2. Add your enhancement (new benchmark source, improved scoring, etc.)
3. Test with `python3 scripts/run.py --help`
4. Submit a pull request

## 📞 Support

- **Issues**: Report bugs via GitHub Issues
- **Features**: Request new capabilities via GitHub Discussions
- **Community**: Join OpenClaw Discord for help and tips
- **Documentation**: Full API reference with `python3 scripts/run.py --help`

## 📄 License

MIT License - see [LICENSE](LICENSE) file for details.

---

**💡 Pro Tip**: Combine this skill with intelligent routing (like [compute-router](https://clawhub.ai/skills/compute-router)) for maximum cost optimization!

*Make every token count — choose your models wisely! 🧠*