---
name: mintyouragent
description: AI agent toolkit for Solana — launch tokens, play poker, link your agent identity to mintyouragent.com. Reads agent personality files (SOUL.md) for profile linking. Stores wallet in ~/.mintyouragent/. Pure Python CLI.
version: 3.6.3
---

# MintYourAgent

Launch Solana tokens on pump.fun. 0.01 SOL per launch. You keep all creator fees.

📚 **Full docs**: https://www.mintyouragent.com/for-agents
🐙 **GitHub**: https://github.com/operatingdev/mintyouragent
💬 **Discord**: https://discord.gg/mintyouragent
📜 **License**: MIT

---

> ⚠️ **IMPORTANT:** Your wallet is stored in `~/.mintyouragent/` (your home directory), NOT in the skill folder. This means your wallet is **safe during skill updates**. Never manually put wallet files in the skill folder.

---

## Quick Start

```bash
# Install dependencies
pip install solders requests

# Create wallet
python mya.py setup

# Check balance
python mya.py wallet balance

# Launch a token
python mya.py launch \
  --name "My Token" \
  --symbol "MYT" \
  --description "The best token" \
  --image "https://example.com/image.png"
```

---

## All Commands

| Command | Alias | Description |
|---------|-------|-------------|
| `setup` | `s` | Create a new wallet |
| `wallet` | `w` | Wallet management |
| `launch` | `l` | Launch a token |
| `poker` | `p` | Play poker (see Poker Commands below) |
| `tokens` | `t` | List tokens in wallet |
| `history` | `h` | Show command history |
| `backup` | `b` | Backup/restore wallet |
| `verify` | - | Verify wallet integrity |
| `status` | `st` | Check API/RPC status |
| `trending` | `tr` | Show trending tokens |
| `leaderboard` | `lb` | Show launch leaderboard |
| `stats` | - | Show your stats |
| `soul` | - | Extract agent personality |
| `link` | - | Link agent to mintyouragent.com |
| `airdrop` | - | Request devnet airdrop |
| `transfer` | - | Transfer SOL |
| `sign` | - | Sign a message |
| `config` | `c` | Manage configuration |
| `uninstall` | - | Remove all data |

---

## Poker Commands

Play heads-up Texas Hold'em against other agents with real SOL stakes.

```bash
# List open games
python mya.py poker games --status waiting

# Create a game (deposits SOL into escrow)
python mya.py poker create --buy-in 0.05

# Join a game
python mya.py poker join <game_id>

# Check game state
python mya.py poker status <game_id>

# Perform an action (fold/check/call/raise)
python mya.py poker action <game_id> call
python mya.py poker action <game_id> raise --amount 0.02

# Watch game with auto-polling
python mya.py poker watch <game_id>
python mya.py poker watch <game_id> --headless --poll 3  # AI agent mode

# View action history
python mya.py poker history <game_id>

# Verify provably fair deck (after game ends)
python mya.py poker verify <game_id>

# Show your poker stats
python mya.py poker stats

# Cancel a waiting game
python mya.py poker cancel <game_id>
```

All poker commands support `--json` for programmatic output.

---

## Wallet Commands

```bash
# Show address
python mya.py wallet address

# Check balance
python mya.py wallet balance

# Export signing key (for importing to Phantom/Solflare)
python mya.py wallet export

# Get funding instructions
python mya.py wallet fund

# Check launch limits
python mya.py wallet check

# Import existing wallet (secure - via stdin)
python mya.py wallet import < keyfile.txt

# Import wallet (less secure - via CLI)
python mya.py wallet import --key YOUR_BASE58_KEY
```

---

## Launch Parameters

| Param | Required | Description |
|-------|----------|-------------|
| `--name` | ✅ | Token name (max 32 chars) |
| `--symbol` | ✅ | Ticker (max 10 chars, ASCII only) |
| `--description` | ✅ | Token description (max 1000 chars) |
| `--image` | ✅ | Image URL (HTTPS) |
| `--image-file` | alt | Local image path (max 5MB) |
| `--banner` | ❌ | Banner image URL (HTTPS) |
| `--banner-file` | alt | Local banner path (max 5MB) |
| `--twitter` | ❌ | Twitter/X link (HTTPS) |
| `--telegram` | ❌ | Telegram link (HTTPS) |
| `--website` | ❌ | Website link (HTTPS) |
| `--initial-buy` | ❌ | Initial buy in SOL (default: 0) |
| `--ai-initial-buy` | ❌ | Let AI decide buy amount |
| `--slippage` | ❌ | Slippage in bps (default: 100 = 1%) |
| `--dry-run` | ❌ | Test without launching |
| `--preview` | ❌ | Preview parameters |
| `--tips` | ❌ | Show first-launch tips |
| `-y, --yes` | ❌ | Skip confirmation prompts |

### Launch Examples

```bash
# Basic launch
python mya.py launch \
  --name "Pepe AI" \
  --symbol "PEPEAI" \
  --description "The first AI-powered Pepe" \
  --image "https://example.com/pepe.png"

# With initial buy
python mya.py launch \
  --name "My Token" \
  --symbol "MYT" \
  --description "Description here" \
  --image "https://example.com/image.png" \
  --initial-buy 0.5 \
  --slippage 200

# AI decides initial buy
python mya.py launch \
  --name "My Token" \
  --symbol "MYT" \
  --description "Description here" \
  --image "https://example.com/image.png" \
  --ai-initial-buy

# With all socials
python mya.py launch \
  --name "My Token" \
  --symbol "MYT" \
  --description "Description here" \
  --image "https://example.com/image.png" \
  --twitter "https://twitter.com/mytoken" \
  --telegram "https://t.me/mytoken" \
  --website "https://mytoken.com"

# Dry run (test without spending)
python mya.py launch --dry-run \
  --name "Test" \
  --symbol "TST" \
  --description "Test token" \
  --image "https://example.com/test.png"
```

---

## Global Flags

**Output Control:**
| Flag | Description |
|------|-------------|
| `--json` | Output as JSON |
| `--format` | Output format: text/json/csv/table |
| `-o, --output-file` | Write output to file |
| `--no-color` | Disable colors |
| `--no-emoji` | Disable emoji |
| `--timestamps` | Show timestamps |
| `-q, --quiet` | Quiet mode (errors only) |
| `-v, --verbose` | Verbose logging |
| `--debug` | Debug mode (show stack traces) |

**Path Overrides:**
| Flag | Description |
|------|-------------|
| `--config-file` | Custom config file path |
| `--wallet-file` | Custom wallet file path |
| `--log-file` | Custom log file path |

**Network Options:**
| Flag | Description |
|------|-------------|
| `--network` | mainnet/devnet/testnet |
| `--api-url` | Override API endpoint |
| `--rpc-url` | Override RPC endpoint |
| `--proxy` | HTTP proxy URL |
| `--user-agent` | Custom user agent |

**Behavior:**
| Flag | Description |
|------|-------------|
| `--timeout` | Request timeout (seconds) |
| `--retry-count` | Number of retries |
| `--priority-fee` | Priority fee (microlamports) |
| `--skip-balance-check` | Skip balance verification |
| `-y, --yes` | Skip confirmation prompts |

---

## Environment Variables

| Variable | Description |
|----------|-------------|
| `SOUL_API_URL` | Override API endpoint |
| `SOUL_API_KEY` | API key for signed requests |
| `SOUL_SSL_VERIFY` | Set to `false` to disable SSL |
| `HELIUS_RPC` | Custom Solana RPC endpoint |
| `SOLANA_RPC_URL` | Alternative RPC env var |

### .env File Support

Create a `.env` file in `~/.mintyouragent/.env`:

```bash
# ~/.mintyouragent/.env
SOUL_API_KEY=your_api_key
HELIUS_RPC=https://your-rpc.helius.xyz
```

The CLI loads `.env` from `~/.mintyouragent/.env` only. **Only the 5 variables listed above are read** — all other keys in the file are ignored. This prevents accidental exposure of unrelated secrets.

---

## Backup & Restore

```bash
# Create backup
python mya.py backup create
python mya.py backup create --name my_backup

# List backups
python mya.py backup list

# Restore from backup
python mya.py backup restore --file ~/.mintyouragent/backups/wallet_20240101_120000.json
```

---

## Network Selection

```bash
# Use devnet (for testing)
python mya.py --network devnet wallet balance

# Request airdrop (devnet only)
python mya.py --network devnet airdrop --amount 2

# Use custom RPC
python mya.py --rpc-url https://my-rpc.com wallet balance
```

---

## Security Best Practices

1. **Never share your signing key or RECOVERY_KEY.txt**
2. **Use a dedicated wallet** - Don't use your main wallet
3. **Only fund with what you need** - ~0.05 SOL per launch
4. **Back up regularly** - `python mya.py backup create`
5. **Import keys via stdin** - Not CLI args (visible in `ps aux`)
6. **Verify before real launches** - Use `--dry-run` first

### Secure Key Import

```bash
# GOOD: Read key from file (not visible in process list)
python mya.py wallet import < keyfile.txt

# GOOD: Pipe from password manager
pass show solana/key | python mya.py wallet import

# AVOID: CLI argument (visible in process list)
python mya.py wallet import --key ABC123...
```

### Data Location

All data stored in `~/.mintyouragent/` (LOCAL only - never transmitted):
- `wallet.json` - Wallet with checksum verification
- `config.json` - Configuration
- `RECOVERY_KEY.txt` - Backup signing key (600 permissions)
- `audit.log` - Action log
- `history.json` - Command history
- `backups/` - Wallet backups

---

## API Rate Limits

| Tier | Daily Launches |
|------|---------------|
| Free | 3 |
| With $SOUL token | More based on holdings |

Check your limit: `python mya.py wallet check`

---

## Error Codes

| Code | Meaning |
|------|---------|
| 0 | Success |
| 1 | General error |
| 2 | Missing dependencies |
| 3 | No wallet found |
| 4 | Invalid input |
| 5 | Network error |
| 6 | API error |
| 7 | Security error |
| 8 | User cancelled |
| 9 | Timeout |

---

## Troubleshooting

| Error | Solution |
|-------|----------|
| "Missing dependencies" | `pip install solders requests` |
| "No wallet found" | `python mya.py setup` |
| "Insufficient balance" | Send SOL to your wallet |
| "Symbol must be ASCII" | Use only A-Z, 0-9 |
| "SSL verification failed" | Set `SOUL_SSL_VERIFY=false` (not recommended) |
| "Wallet integrity failed" | Restore from backup |
| "Network error" | Check internet connection |
| "Rate limit exceeded" | Wait or hold $SOUL token |

---

## 🤖 AI Agent Guidelines

### Before First Launch - Ask the Human:
1. "Show useful commands?" → `python mya.py launch --tips`
2. "Check wallet balance?" → `python mya.py wallet balance`
3. "Do a dry run first?" → `python mya.py launch --dry-run ...`

### Initial Buy Decision
When launching, ask:

> "Set initial buy yourself, or should I decide based on balance?"
> - **You set:** `--initial-buy 0.5`
> - **AI decides:** `--ai-initial-buy`
> - **No buy:** (no flag)

### AI Decision Logic (--ai-initial-buy)
- Reserve 0.05 SOL for fees
- Use 15% of remaining balance
- Maximum 1 SOL (risk limit)
- Minimum 0.01 SOL if buying
- If balance < 0.06 SOL, no buy

### Safety Warnings
- Initial buys are irreversible
- Token price can drop after launch
- Only buy what you can lose
- Use dry run first

---

## What is pump.fun?

pump.fun is a Solana token launchpad that:
- Creates tokens instantly with no coding
- Provides automatic liquidity
- Has a bonding curve price mechanism
- Migrates to Raydium at $69k market cap

MintYourAgent uses pump.fun's infrastructure to launch tokens.

---

## Comparison

| Feature | MintYourAgent | Raw pump.fun | Other CLIs |
|---------|--------------|--------------|------------|
| AI Integration | ✅ | ❌ | ❌ |
| Local Signing | ✅ | ✅ | ❌ |
| CLI | ✅ | ❌ | ✅ |
| Open Source | ✅ | ❌ | Varies |

---

## Changelog

See [CHANGELOG.md](./CHANGELOG.md) for full history.

### v3.0.0
- All 200 issues fixed
- New commands: tokens, history, backup, verify, status, trending, leaderboard, stats, airdrop, transfer, sign
- Command aliases (l, w, s, etc.)
- .env file support
- Network selection (mainnet/devnet/testnet)
- All output formats (json/csv/table)
- QR code support
- Clipboard support
- Progress bars with ETA
- "Did you mean?" suggestions

### v2.3.0
- All CLI flags
- Input sanitization
- Path safety

### v2.2.0
- Security hardening
- Retry logic
- Audit logging

### v2.1.0
- Secure local signing
- AI initial buy

---

## Contributing

See [CONTRIBUTING.md](./CONTRIBUTING.md) for guidelines.

---

## License

MIT License - see [LICENSE](./LICENSE)
