#!/usr/bin/env python3
"""
Fetch planetary positions from astronomyapi.com and output zodiac transit data.
Usage: python fetch-planetary-positions.py --lat 10.82 --lon 106.63
Env: ASTRONOMY_APP_ID, ASTRONOMY_APP_SECRET
"""

import argparse
import base64
import json
import os
import sys
from datetime import datetime, timedelta, timezone
from urllib.request import urlopen, Request
from urllib.error import URLError

API_BASE = "https://api.astronomyapi.com/api/v2"
SIGNS = [
    "Aries", "Taurus", "Gemini", "Cancer", "Leo", "Virgo",
    "Libra", "Scorpio", "Sagittarius", "Capricorn", "Aquarius", "Pisces",
]
PLANETS = ["sun", "moon", "mercury", "venus", "mars", "jupiter", "saturn", "uranus", "neptune", "pluto"]
ASPECTS = {"conjunction": 0, "sextile": 60, "square": 90, "trine": 120, "opposition": 180}
ASPECT_ORBS = {"conjunction": 8, "sextile": 6, "square": 7, "trine": 8, "opposition": 8}


def get_auth_header():
    """Build Basic Auth header from env vars."""
    app_id = os.environ.get("ASTRONOMY_APP_ID", "")
    app_secret = os.environ.get("ASTRONOMY_APP_SECRET", "")
    if not app_id or not app_secret:
        return None
    token = base64.b64encode(f"{app_id}:{app_secret}".encode()).decode()
    return f"Basic {token}"


def longitude_to_sign(lon_deg):
    """Convert ecliptic longitude to zodiac sign + degree."""
    sign_idx = int(lon_deg / 30) % 12
    degree_in_sign = round(lon_deg % 30, 1)
    return SIGNS[sign_idx], degree_in_sign


def fetch_positions(date_str, lat, lon, auth):
    """Fetch body positions for a given date."""
    url = (
        f"{API_BASE}/bodies/positions"
        f"?latitude={lat}&longitude={lon}&elevation=0"
        f"&from_date={date_str}&to_date={date_str}&time=12:00:00"
    )
    req = Request(url, headers={"Authorization": auth})
    try:
        with urlopen(req, timeout=15) as resp:
            return json.loads(resp.read().decode())
    except (URLError, json.JSONDecodeError) as e:
        print(json.dumps({"error": str(e), "fallback": True}), file=sys.stderr)
        return None


def parse_planet_data(data):
    """Extract planet positions from API response. Returns dict of name -> ecliptic longitude."""
    positions = {}
    if not data or "data" not in data:
        return positions
    rows = data["data"].get("table", {}).get("rows", [])
    for row in rows:
        body_id = row.get("entry", {}).get("id", "").lower()
        if body_id not in PLANETS:
            continue
        cells = row.get("cells", [])
        if not cells:
            continue
        # Use first cell (single date query)
        cell = cells[0]
        # Try ecliptic longitude first, fall back to RA conversion
        ecl = cell.get("position", {}).get("ecliptic", {})
        lon = ecl.get("longitude", {})
        if isinstance(lon, dict):
            # Format: {"degrees": "315", "minutes": "23", "seconds": "45"}
            deg = float(lon.get("degrees", 0))
            mins = float(lon.get("minutes", 0))
            secs = float(lon.get("seconds", 0))
            positions[body_id] = deg + mins / 60 + secs / 3600
        elif isinstance(lon, (int, float)):
            positions[body_id] = float(lon)
        else:
            # Fallback: try equatorial RA and approximate (rough)
            eq = cell.get("position", {}).get("equatorial", {})
            ra_h = float(eq.get("rightAscension", {}).get("hours", 0))
            # Very rough RA->ecliptic: ecliptic_lon ~ RA_degrees (ignores obliquity)
            positions[body_id] = (ra_h * 15) % 360
    return positions


def detect_aspects(positions):
    """Detect major aspects between planet pairs."""
    aspects_found = []
    planet_names = list(positions.keys())
    for i in range(len(planet_names)):
        for j in range(i + 1, len(planet_names)):
            p1, p2 = planet_names[i], planet_names[j]
            diff = abs(positions[p1] - positions[p2])
            if diff > 180:
                diff = 360 - diff
            for aspect_name, angle in ASPECTS.items():
                orb = abs(diff - angle)
                if orb <= ASPECT_ORBS[aspect_name]:
                    aspects_found.append({
                        "planet1": p1.capitalize(), "planet2": p2.capitalize(),
                        "aspect": aspect_name, "orb": round(orb, 1),
                    })
    return aspects_found


def build_output(today_pos, yesterday_pos):
    """Build final JSON output with signs, retrograde, aspects."""
    planets = []
    for name in PLANETS:
        if name not in today_pos:
            continue
        lon = today_pos[name]
        sign, degree = longitude_to_sign(lon)
        retrograde = False
        if name in yesterday_pos and name not in ("sun", "moon"):
            delta = today_pos[name] - yesterday_pos[name]
            # Handle 360 wraparound
            if delta > 180:
                delta -= 360
            elif delta < -180:
                delta += 360
            retrograde = delta < 0
        planets.append({
            "name": name.capitalize(), "sign": sign,
            "degree": degree, "retrograde": retrograde,
        })

    # Moon phase approximation from elongation (Sun-Moon angle)
    moon_info = {}
    if "moon" in today_pos and "sun" in today_pos:
        elong = (today_pos["moon"] - today_pos["sun"]) % 360
        sign, degree = longitude_to_sign(today_pos["moon"])
        phase = _moon_phase_name(elong)
        moon_info = {"phase": phase, "illumination": round(abs(180 - elong) / 180, 2), "sign": sign}

    return {
        "date": datetime.now(timezone.utc).strftime("%Y-%m-%d"),
        "planets": planets,
        "moon": moon_info,
        "aspects": detect_aspects(today_pos),
    }


def _moon_phase_name(elong):
    """Approximate moon phase name from Sun-Moon elongation."""
    phases = ["New Moon", "Waxing Crescent", "First Quarter", "Waxing Gibbous",
              "Full Moon", "Waning Gibbous", "Last Quarter", "Waning Crescent"]
    return phases[int(elong / 45) % 8]


def main():
    parser = argparse.ArgumentParser(description="Fetch planetary positions for horoscope")
    parser.add_argument("--lat", type=float, default=0.0, help="Observer latitude")
    parser.add_argument("--lon", type=float, default=0.0, help="Observer longitude")
    args = parser.parse_args()

    auth = get_auth_header()
    if not auth:
        print(json.dumps({"error": "Missing ASTRONOMY_APP_ID or ASTRONOMY_APP_SECRET", "fallback": True}))
        sys.exit(1)

    today = datetime.now(timezone.utc).strftime("%Y-%m-%d")
    yesterday = (datetime.now(timezone.utc) - timedelta(days=1)).strftime("%Y-%m-%d")

    today_data = fetch_positions(today, args.lat, args.lon, auth)
    yesterday_data = fetch_positions(yesterday, args.lat, args.lon, auth)

    today_pos = parse_planet_data(today_data) if today_data else {}
    yesterday_pos = parse_planet_data(yesterday_data) if yesterday_data else {}

    if not today_pos:
        print(json.dumps({"error": "No planetary data retrieved", "fallback": True, "date": today}))
        sys.exit(1)

    result = build_output(today_pos, yesterday_pos)
    print(json.dumps(result, indent=2))


if __name__ == "__main__":
    main()
