#!/usr/bin/env python3
"""
I Ching hexagram casting — random (3-coin method) or manual input.
Usage:
  Random: python cast-i-ching-hexagram.py --mode random
  Manual: python cast-i-ching-hexagram.py --mode manual --upper Kan --lower Kun --moving 2,1
"""
import argparse
import json
import secrets
import sys
from pathlib import Path

SCRIPT_DIR = Path(__file__).parent
HEXAGRAM_FILE = SCRIPT_DIR.parent / "references" / "i-ching-64-hexagrams.json"

# 8 trigrams: name -> binary (bottom to top: line1, line2, line3)
# Yang=1, Yin=0
TRIGRAMS = {
    "Qian": (1, 1, 1), "Dui": (1, 1, 0), "Li": (1, 0, 1), "Zhen": (1, 0, 0),
    "Xun": (0, 1, 1), "Kan": (0, 1, 0), "Gen": (0, 0, 1), "Kun": (0, 0, 0),
}
# Reverse: binary tuple -> trigram name
BINARY_TO_TRIGRAM = {v: k for k, v in TRIGRAMS.items()}


def load_hexagrams():
    """Load 64 hexagrams from JSON file."""
    with open(HEXAGRAM_FILE, "r", encoding="utf-8") as f:
        data = json.load(f)
    # Index by (upper, lower) trigram pair
    by_pair = {}
    by_num = {}
    for h in data:
        by_pair[(h["upper"], h["lower"])] = h
        by_num[h["num"]] = h
    return by_pair, by_num


def coin_toss_line():
    """Three-coin method: toss 3 coins, return (value, is_moving).
    Heads=3, Tails=2. Sum: 6=old yin(moving), 7=young yang, 8=young yin, 9=old yang(moving).
    """
    total = sum(3 if secrets.randbelow(2) else 2 for _ in range(3))
    yang = total in (7, 9)  # yang line
    moving = total in (6, 9)  # changing line
    return int(yang), moving


def cast_random():
    """Cast 6 lines using 3-coin method. Returns (lines, moving_positions)."""
    lines = []
    moving = []
    for i in range(6):
        yang, is_moving = coin_toss_line()
        lines.append(yang)
        if is_moving:
            moving.append(i + 1)  # 1-indexed
    return lines, moving


def lines_to_trigrams(lines):
    """Convert 6 lines to (lower_trigram, upper_trigram) names."""
    lower = tuple(lines[0:3])
    upper = tuple(lines[3:6])
    lower_name = BINARY_TO_TRIGRAM.get(lower, "Unknown")
    upper_name = BINARY_TO_TRIGRAM.get(upper, "Unknown")
    return upper_name, lower_name


def apply_moving_lines(lines, moving_positions):
    """Flip moving lines to get transformed hexagram."""
    transformed = list(lines)
    for pos in moving_positions:
        idx = pos - 1  # convert to 0-indexed
        transformed[idx] = 1 - transformed[idx]
    return transformed


def build_spark(primary, transformed, moving_positions):
    """Generate SPARK summary: Status/Problem/Advantage/Action/Prospect."""
    return {
        "status": primary.get("keyword", ""),
        "problem": f"Moving lines at {moving_positions}" if moving_positions else "No moving lines — stable",
        "advantage": primary.get("meaning", ""),
        "action": f"Transition toward {transformed['keyword']}" if transformed else "Maintain current course",
        "prospect": transformed.get("meaning", primary.get("meaning", "")),
    }


def main():
    parser = argparse.ArgumentParser(description="I Ching hexagram casting")
    parser.add_argument("--mode", choices=["random", "manual"], default="random")
    parser.add_argument("--upper", help="Upper trigram name (manual mode)")
    parser.add_argument("--lower", help="Lower trigram name (manual mode)")
    parser.add_argument("--moving", help="Moving line positions, comma-separated (e.g., 2,1)")
    args = parser.parse_args()

    by_pair, by_num = load_hexagrams()

    if args.mode == "manual":
        if not args.upper or not args.lower:
            print(json.dumps({"error": "Manual mode requires --upper and --lower trigrams"}))
            sys.exit(1)
        upper, lower = args.upper, args.lower
        if upper not in TRIGRAMS or lower not in TRIGRAMS:
            print(json.dumps({"error": f"Unknown trigram. Valid: {list(TRIGRAMS.keys())}"}))
            sys.exit(1)
        # Build lines from trigrams
        lines = list(TRIGRAMS[lower]) + list(TRIGRAMS[upper])
        moving_positions = [int(x) for x in args.moving.split(",")] if args.moving else []
    else:
        lines, moving_positions = cast_random()
        upper, lower = lines_to_trigrams(lines)

    # Lookup primary hexagram
    primary = by_pair.get((upper, lower))
    if not primary:
        print(json.dumps({"error": f"Hexagram not found for {upper}/{lower}"}))
        sys.exit(1)

    # Calculate transformed hexagram
    transformed = None
    if moving_positions:
        t_lines = apply_moving_lines(lines, moving_positions)
        t_upper, t_lower = lines_to_trigrams(t_lines)
        transformed = by_pair.get((t_upper, t_lower))

    result = {
        "primary": {
            "number": primary["num"], "name": primary["name"],
            "viet": primary["viet"], "meaning": primary["meaning"],
            "upper_trigram": upper, "lower_trigram": lower,
            "keyword": primary["keyword"],
        },
        "moving_lines": moving_positions,
        "transformed": {
            "number": transformed["num"], "name": transformed["name"],
            "viet": transformed["viet"], "meaning": transformed["meaning"],
            "upper_trigram": transformed["upper"], "lower_trigram": transformed["lower"],
            "keyword": transformed["keyword"],
        } if transformed else None,
        "spark": build_spark(primary, transformed or primary, moving_positions),
    }
    print(json.dumps(result, ensure_ascii=False, indent=2))


if __name__ == "__main__":
    main()
