#!/usr/bin/env python3
"""
Calculate Western natal chart using kerykeion library.
Outputs JSON format compatible with Tiger's structure.
"""

import argparse
import json
import sys
from datetime import datetime
from typing import Dict, List, Any

try:
    from kerykeion import AstrologicalSubject, NatalAspects
except ImportError as e:
    print(json.dumps({"error": f"kerykeion not installed: {e}", "fallback": True}))
    sys.exit(1)


def calculate_aspects(subject: AstrologicalSubject) -> List[Dict[str, Any]]:
    """Calculate aspects between planets."""
    aspects_list = []

    try:
        # Use kerykeion's aspect calculator
        natal_aspects = NatalAspects(subject)

        # Map aspect types
        aspect_map = {
            0: "conjunction",
            60: "sextile",
            90: "square",
            120: "trine",
            180: "opposition"
        }

        # Extract aspects from kerykeion
        for aspect in natal_aspects.all_aspects:
            aspect_type = aspect_map.get(aspect["aspect_degrees"], "unknown")
            aspects_list.append({
                "from": aspect["p1_name"],
                "to": aspect["p2_name"],
                "type": aspect_type,
                "orb": abs(aspect["orbit"])
            })
    except Exception as e:
        # Fallback: manual aspect calculation
        planets = [
            ("Sun", subject.sun),
            ("Moon", subject.moon),
            ("Mercury", subject.mercury),
            ("Venus", subject.venus),
            ("Mars", subject.mars),
            ("Jupiter", subject.jupiter),
            ("Saturn", subject.saturn),
            ("Uranus", subject.uranus),
            ("Neptune", subject.neptune),
            ("Pluto", subject.pluto)
        ]

        # Calculate aspects manually
        for i, (name1, planet1) in enumerate(planets):
            for name2, planet2 in planets[i+1:]:
                lon1 = planet1.abs_pos
                lon2 = planet2.abs_pos

                angle = abs(lon1 - lon2)
                if angle > 180:
                    angle = 360 - angle

                # Check major aspects with orbs
                aspect_checks = [
                    (0, 8, "conjunction"),
                    (60, 6, "sextile"),
                    (90, 8, "square"),
                    (120, 8, "trine"),
                    (180, 8, "opposition")
                ]

                for aspect_angle, orb_limit, aspect_type in aspect_checks:
                    orb = abs(angle - aspect_angle)
                    if orb <= orb_limit:
                        aspects_list.append({
                            "from": name1,
                            "to": name2,
                            "type": aspect_type,
                            "orb": round(orb, 2)
                        })

    return aspects_list


SIGN_FULL = {"Ari": "Aries", "Tau": "Taurus", "Gem": "Gemini", "Can": "Cancer",
             "Leo": "Leo", "Vir": "Virgo", "Lib": "Libra", "Sco": "Scorpio",
             "Sag": "Sagittarius", "Cap": "Capricorn", "Aqu": "Aquarius", "Pis": "Pisces"}
HOUSE_ATTRS = ["first_house", "second_house", "third_house", "fourth_house",
               "fifth_house", "sixth_house", "seventh_house", "eighth_house",
               "ninth_house", "tenth_house", "eleventh_house", "twelfth_house"]


def full_sign(abbr):
    return SIGN_FULL.get(abbr, abbr)


def get_planet_data(planet_obj) -> Dict[str, Any]:
    """Extract planet data in required format."""
    return {
        "lon": round(planet_obj.abs_pos, 2),
        "sign": full_sign(planet_obj.sign),
        "degree_in_sign": round(planet_obj.position, 2),
        "house": planet_obj.house,
        "retrograde": planet_obj.retrograde
    }


def calculate_natal_chart(date_str, time_str, timezone, lat, lon, name):
    """Calculate full natal chart."""
    try:
        dt = datetime.strptime(f"{date_str} {time_str}", "%Y-%m-%d %H:%M")
        subject = AstrologicalSubject(
            name=name, year=dt.year, month=dt.month, day=dt.day,
            hour=dt.hour, minute=dt.minute, city="", lat=lat, lng=lon, tz_str=timezone
        )

        # House cusps from named attributes
        cusps = [round(getattr(subject, h).abs_pos, 2) for h in HOUSE_ATTRS]

        # Planets
        planet_names = ["sun", "moon", "mercury", "venus", "mars",
                        "jupiter", "saturn", "uranus", "neptune", "pluto"]
        bodies = {}
        for pname in planet_names:
            obj = getattr(subject, pname)
            bodies[pname.capitalize()] = get_planet_data(obj)

        # Lunar node (v5 API)
        node = getattr(subject, "mean_north_lunar_node", None)
        if node is None:
            node = getattr(subject, "mean_node", None)
        if node:
            bodies["North Node"] = get_planet_data(node)

        aspects = calculate_aspects(subject)

        return {
            "meta": {"zodiac": "tropical", "house_system": subject.houses_system_name},
            "geo": {"lat": lat, "lon": lon},
            "houses": {
                "asc": round(subject.first_house.abs_pos, 2),
                "mc": round(subject.tenth_house.abs_pos, 2),
                "cusps": cusps,
            },
            "bodies": bodies,
            "aspects": aspects,
        }
    except Exception as e:
        return {"error": str(e), "fallback": True}


def main():
    parser = argparse.ArgumentParser(description="Calculate Western natal chart")
    parser.add_argument("--date", required=True, help="Birth date (YYYY-MM-DD)")
    parser.add_argument("--time", required=True, help="Birth time (HH:MM)")
    parser.add_argument("--tz", required=True, help="Timezone (e.g., 'Asia/Saigon')")
    parser.add_argument("--lat", required=True, type=float, help="Latitude")
    parser.add_argument("--lon", required=True, type=float, help="Longitude")
    parser.add_argument("--name", required=True, help="Person's name")

    args = parser.parse_args()

    result = calculate_natal_chart(
        args.date,
        args.time,
        args.tz,
        args.lat,
        args.lon,
        args.name
    )

    print(json.dumps(result, indent=2))


if __name__ == "__main__":
    main()
