#!/usr/bin/env python3
"""
Numerology calculations script (stdlib only).
Calculates Life Path, Birthday, Attitude, Challenges, Pinnacles, and Personal cycles.
"""

import argparse
import json
import sys
from datetime import datetime


def reduce(n):
    """
    Reduce number to single digit or master number (11, 22, 33).
    Example: 1990 → 1+9+9+0=19 → 1+9=10 → 1+0=1
    """
    if n in (11, 22, 33):
        return n

    while n > 9:
        n = sum(int(digit) for digit in str(n))
        if n in (11, 22, 33):
            return n

    return n


def calculate_life_path(year, month, day):
    """
    Life Path: reduce(full birth date digits).
    Example: 2000-03-25 → 2+0+0+0+0+3+2+5=12 → 1+2=3
    """
    total = sum(int(d) for d in f"{year:04d}{month:02d}{day:02d}")
    return reduce(total)


def calculate_birthday(day):
    """
    Birthday: reduce(day).
    Example: 25 → 2+5=7
    """
    return reduce(day)


def calculate_attitude(month, day):
    """
    Attitude: reduce(month + day).
    Example: 3+25=28 → 2+8=10 → 1+0=1
    """
    return reduce(month + day)


def calculate_challenges(year, month, day):
    """
    Challenges (4 levels):
    - C1 = abs(reduce(month) - reduce(day))
    - C2 = abs(reduce(day) - reduce(year))
    - C3 = abs(C1 - C2)
    - C4 = abs(reduce(month) - reduce(year))
    """
    rm = reduce(month)
    rd = reduce(day)
    ry = reduce(year)

    c1 = abs(rm - rd)
    c2 = abs(rd - ry)
    c3 = abs(c1 - c2)
    c4 = abs(rm - ry)

    return [c1, c2, c3, c4]


def calculate_pinnacles(year, month, day):
    """
    Pinnacles (4 levels):
    - P1 = reduce(month + day)
    - P2 = reduce(day + year)
    - P3 = reduce(P1 + P2)
    - P4 = reduce(month + year)
    """
    p1 = reduce(month + day)
    p2 = reduce(day + year)
    p3 = reduce(p1 + p2)
    p4 = reduce(month + year)

    return [p1, p2, p3, p4]


def calculate_personal_cycles(birth_month, birth_day, current_date):
    """
    Calculate Personal Year, Month, and Day.
    - Personal Year: reduce(birth_month + birth_day + current_year)
    - Personal Month: reduce(personal_year + current_month)
    - Personal Day: reduce(personal_month + current_day)
    """
    current_year = current_date.year
    current_month = current_date.month
    current_day = current_date.day

    personal_year = reduce(birth_month + birth_day + current_year)
    personal_month = reduce(personal_year + current_month)
    personal_day = reduce(personal_month + current_day)

    return personal_year, personal_month, personal_day


def get_number_meaning(number):
    """Get keyword meaning for numerology number."""
    meanings = {
        1: "Leadership",
        2: "Partnership",
        3: "Creativity",
        4: "Foundation",
        5: "Change",
        6: "Harmony",
        7: "Wisdom",
        8: "Power",
        9: "Completion",
        11: "Intuition",
        22: "Master Builder",
        33: "Master Teacher"
    }
    return meanings.get(number, "Unknown")


def main():
    parser = argparse.ArgumentParser(
        description="Calculate numerology numbers from birth date"
    )
    parser.add_argument(
        "--date",
        required=True,
        help="Birth date in YYYY-MM-DD format"
    )

    args = parser.parse_args()

    try:
        birth_date = datetime.strptime(args.date, "%Y-%m-%d")
    except ValueError:
        print(f"Error: Invalid date format. Use YYYY-MM-DD", file=sys.stderr)
        sys.exit(1)

    year = birth_date.year
    month = birth_date.month
    day = birth_date.day

    # Calculate core numbers
    life_path = calculate_life_path(year, month, day)
    birthday = calculate_birthday(day)
    attitude = calculate_attitude(month, day)
    challenges = calculate_challenges(year, month, day)
    pinnacles = calculate_pinnacles(year, month, day)

    # Calculate personal cycles (using today's date)
    today = datetime.now()
    personal_year, personal_month, personal_day = calculate_personal_cycles(
        month, day, today
    )

    # Build result
    result = {
        "birth_date": args.date,
        "life_path": {
            "number": life_path,
            "meaning": get_number_meaning(life_path)
        },
        "birthday": {
            "number": birthday,
            "meaning": get_number_meaning(birthday)
        },
        "attitude": {
            "number": attitude,
            "meaning": get_number_meaning(attitude)
        },
        "challenges": challenges,
        "pinnacles": pinnacles,
        "personal_year": {
            "number": personal_year,
            "meaning": get_number_meaning(personal_year)
        },
        "personal_month": {
            "number": personal_month,
            "meaning": get_number_meaning(personal_month)
        },
        "personal_day": {
            "number": personal_day,
            "meaning": get_number_meaning(personal_day)
        }
    }

    print(json.dumps(result, indent=2))


if __name__ == "__main__":
    main()
