#!/usr/bin/env python3
"""Ba-Zi (Four Pillars) Calculator - Western zodiac & Ba-Zi chart from birth date/time."""
import argparse
import json
from datetime import datetime
from zoneinfo import ZoneInfo

# 10 Heavenly Stems
STEMS = [
    {"zh": "甲", "viet": "Giáp", "element": "Wood", "polarity": "Yang"},
    {"zh": "乙", "viet": "Ất", "element": "Wood", "polarity": "Yin"},
    {"zh": "丙", "viet": "Bính", "element": "Fire", "polarity": "Yang"},
    {"zh": "丁", "viet": "Đinh", "element": "Fire", "polarity": "Yin"},
    {"zh": "戊", "viet": "Mậu", "element": "Earth", "polarity": "Yang"},
    {"zh": "己", "viet": "Kỷ", "element": "Earth", "polarity": "Yin"},
    {"zh": "庚", "viet": "Canh", "element": "Metal", "polarity": "Yang"},
    {"zh": "辛", "viet": "Tân", "element": "Metal", "polarity": "Yin"},
    {"zh": "壬", "viet": "Nhâm", "element": "Water", "polarity": "Yang"},
    {"zh": "癸", "viet": "Quý", "element": "Water", "polarity": "Yin"},
]
# 12 Earthly Branches
BRANCHES = [
    {"zh": "子", "viet": "Tý", "animal": "Rat", "element": "Water"},
    {"zh": "丑", "viet": "Sửu", "animal": "Ox", "element": "Earth"},
    {"zh": "寅", "viet": "Dần", "animal": "Tiger", "element": "Wood"},
    {"zh": "卯", "viet": "Mão", "animal": "Rabbit", "element": "Wood"},
    {"zh": "辰", "viet": "Thìn", "animal": "Dragon", "element": "Earth"},
    {"zh": "巳", "viet": "Tỵ", "animal": "Snake", "element": "Fire"},
    {"zh": "午", "viet": "Ngọ", "animal": "Horse", "element": "Fire"},
    {"zh": "未", "viet": "Mùi", "animal": "Goat", "element": "Earth"},
    {"zh": "申", "viet": "Thân", "animal": "Monkey", "element": "Metal"},
    {"zh": "酉", "viet": "Dậu", "animal": "Rooster", "element": "Metal"},
    {"zh": "戌", "viet": "Tuất", "animal": "Dog", "element": "Earth"},
    {"zh": "亥", "viet": "Hợi", "animal": "Pig", "element": "Water"},
]
# Western Zodiac signs (tropical dates)
WESTERN_SIGNS = [
    ("Capricorn", (12, 22), (1, 19)), ("Aquarius", (1, 20), (2, 18)),
    ("Pisces", (2, 19), (3, 20)), ("Aries", (3, 21), (4, 19)),
    ("Taurus", (4, 20), (5, 20)), ("Gemini", (5, 21), (6, 20)),
    ("Cancer", (6, 21), (7, 22)), ("Leo", (7, 23), (8, 22)),
    ("Virgo", (8, 23), (9, 22)), ("Libra", (9, 23), (10, 22)),
    ("Scorpio", (10, 23), (11, 21)), ("Sagittarius", (11, 22), (12, 21)),
]

def get_western_sign(month, day):
    for sign, (start_m, start_d), (end_m, end_d) in WESTERN_SIGNS:
        if start_m == end_m and month == start_m and start_d <= day <= end_d:
            return sign
        elif (month == start_m and day >= start_d) or (month == end_m and day <= end_d):
            return sign
    return "Capricorn"

def julian_day_number(year, month, day):
    a = (14 - month) // 12
    y = year + 4800 - a
    m = month + 12 * a - 3
    return day + (153 * m + 2) // 5 + 365 * y + y // 4 - y // 100 + y // 400 - 32045

def calculate_year_pillar(year):
    return (year - 4) % 10, (year - 4) % 12

def calculate_month_pillar(year_stem_idx, month):
    month_to_branch = {2:2, 3:3, 4:4, 5:5, 6:6, 7:7, 8:8, 9:9, 10:10, 11:11, 12:0, 1:1}
    branch_idx = month_to_branch[month]
    base_stem = [2, 4, 6, 8, 0][year_stem_idx % 5]
    stem_idx = (base_stem + branch_idx - 2) % 10
    return stem_idx, branch_idx

def calculate_day_pillar(year, month, day):
    jdn = julian_day_number(year, month, day)
    return (jdn + 9) % 10, (jdn + 1) % 12

def calculate_hour_pillar(day_stem_idx, hour, minute):
    branch_idx = ((hour + 1) // 2) % 12
    base_stem = [0, 2, 4, 6, 8][day_stem_idx % 5]
    return (base_stem + branch_idx) % 10, branch_idx

def format_pillar(stem_idx, branch_idx):
    stem, branch = STEMS[stem_idx], BRANCHES[branch_idx]
    return {"stem": stem["zh"], "branch": branch["zh"],
            "viet": f"{stem['viet']} {branch['viet']}",
            "english": f"{stem['element']} {branch['animal']}"}

def calculate_five_elements(year_stem, year_branch, month_stem, month_branch,
                           day_stem, day_branch, hour_stem, hour_branch):
    elements = {"Wood": 0, "Fire": 0, "Earth": 0, "Metal": 0, "Water": 0}
    for stem_idx in [year_stem, month_stem, day_stem, hour_stem]:
        elements[STEMS[stem_idx]["element"]] += 1
    for branch_idx in [year_branch, month_branch, day_branch, hour_branch]:
        elements[BRANCHES[branch_idx]["element"]] += 1
    return elements

def analyze_elements(elements):
    lacking = [e for e, count in elements.items() if count == 0]
    strong = [e for e, count in elements.items() if count >= 3]
    parts = []
    if lacking: parts.append(f"Lacks {', '.join(lacking)}")
    if strong: parts.append(f"Strong in {', '.join(strong)}")
    return ". ".join(parts) if parts else "Balanced elements."

def calculate_bazi(date_str, time_str, tz_str):
    tz = ZoneInfo(tz_str)
    dt = datetime.fromisoformat(f"{date_str}T{time_str}").replace(tzinfo=tz)
    year, month, day, hour, minute = dt.year, dt.month, dt.day, dt.hour, dt.minute

    western_sign = get_western_sign(month, day)
    year_stem, year_branch = calculate_year_pillar(year)
    month_stem, month_branch = calculate_month_pillar(year_stem, month)
    day_stem, day_branch = calculate_day_pillar(year, month, day)
    hour_stem, hour_branch = calculate_hour_pillar(day_stem, hour, minute)

    bazi_chart = {"year": format_pillar(year_stem, year_branch),
                  "month": format_pillar(month_stem, month_branch),
                  "day": format_pillar(day_stem, day_branch),
                  "hour": format_pillar(hour_stem, hour_branch)}

    day_master_stem = STEMS[day_stem]
    day_master = {"chinese": day_master_stem["zh"], "viet": day_master_stem["viet"],
                  "element": day_master_stem["element"], "polarity": day_master_stem["polarity"]}

    elements = calculate_five_elements(year_stem, year_branch, month_stem, month_branch,
                                       day_stem, day_branch, hour_stem, hour_branch)

    return {"western_sign": western_sign, "bazi_chart": bazi_chart,
            "day_master": day_master, "five_elements": elements,
            "element_analysis": analyze_elements(elements)}

def main():
    parser = argparse.ArgumentParser(description="Calculate Ba-Zi chart")
    parser.add_argument("--date", required=True, help="Birth date (YYYY-MM-DD)")
    parser.add_argument("--time", required=True, help="Birth time (HH:MM)")
    parser.add_argument("--tz", required=True, help="Timezone (e.g., Asia/Saigon)")
    args = parser.parse_args()
    try:
        result = calculate_bazi(args.date, args.time, args.tz)
        print(json.dumps(result, ensure_ascii=False, indent=2))
    except Exception as e:
        print(json.dumps({"error": str(e)}, ensure_ascii=False))
        exit(1)

if __name__ == "__main__":
    main()
