---
name: mindgraph
description: "A high-performance, structured knowledge graph memory system for AI agents. Provides 18 cognitive tools for reasoning, decision tracking, and goal management. PRIVACY: The server is fully local. The optional extraction scripts read session transcripts and ~/.openclaw/openclaw.json (for API keys), then POST summaries to your configured LLM provider (Gemini/Anthropic/Moonshot). Extraction and dreaming are opt-in. See SKILL.md Security section and INTEGRATION.md."
---

# MindGraph

MindGraph is a **structured knowledge graph index** that transforms AI memory from simple text files into a traversable, evidence-backed cognitive layer. It runs entirely on your machine using a high-performance Rust server, with a strict 28-node ontology and built-in vector search.

---

## ⚠️ Security & Privacy Disclosure

**Read this before enabling extraction or dreaming.**

MindGraph has two distinct components with very different privacy profiles:

### 1. The Server (local-only, no external calls)
The `mindgraph-server` binary stores and queries your knowledge graph entirely on-device. It makes no external network calls. Graph data never leaves your machine.

### 2. The Extraction Scripts (reads + transmits)
`extract.js` and the heartbeat extraction flow **do** read your conversation transcripts and send excerpts to an LLM provider for summarization and node extraction. Specifically:

- **Reads**: Session JSONL transcripts from `agents/main/sessions/`
- **Reads**: `~/.openclaw/openclaw.json` to find your model provider API keys
- **Transmits**: Conversation summaries/excerpts to your configured LLM provider (Gemini, Moonshot, or Anthropic — whichever is in your OpenClaw config)

The extraction script uses the **same provider you already trust for your agent's conversations**. It does not introduce a new third party. However, you should be aware that conversation content is transmitted to that provider during extraction.

### What is NOT transmitted
- Your full raw transcripts (the script flattens and optionally summarizes before sending)
- Your graph database contents
- Any data to ClawHub, MindGraph maintainers, or any party other than your configured LLM provider

### Opting out of extraction
The server works fully without extraction. You can use the 18 cognitive tools manually (via `mindgraph-client.js`) without ever running `extract.js`. Heartbeat extraction and nightly dreaming are opt-in steps described in INTEGRATION.md.

---

## Core Truths

1. **Text is Canonical; Graph is Logic**: Your Markdown files (MEMORY.md, logs) remain the source of truth. MindGraph provides the **relational logic** and **fast retrieval** on top of them.
2. **Tools > Nodes**: Never "add nodes" manually. Use the 18 cognitive tools (Arguments, Inquiries, Commitments) to ensure the graph maintains strict logical integrity.
3. **Epistemic Separation**: Always separate **speculation** from **verified facts**. Use the discovery-verification pattern to prevent graph contamination.

## Design Conventions

1. **Atomic Bundling**: Use bundle endpoints (`mg.addArgument`, `mg.procedure`, `mg.plan`) to create related nodes and edges in a single transaction.
2. **Session Framing**: Call `mg.sessionOp({ action: 'open' })` at the start of each session and `mg.sessionOp({ action: 'trace' })` to record key discoveries.
3. **Opt-in Extraction**: Heartbeat extraction is a recommended workflow, not automatic. Enable it intentionally in your `HEARTBEAT.md`.

## The 18 Cognitive Tools

### Reality Layer
- `mg.ingest`: Capture sources, snippets, or observations.
- `mg.manageEntity`: Create, alias, or merge entities (Person, Org, etc.).

### Epistemic Layer (Reasoning)
- `mg.addArgument`: Atomic Toulmin bundle (Claim + Evidence + Warrant).
- `mg.addInquiry`: Record hypotheses, anomalies, and questions.
- `mg.addStructure`: Crystallize concepts, patterns, and models.

### Intent Layer (Commitments)
- `mg.addCommitment`: Declare goals, projects, or milestones.
- `mg.deliberate`: Open decisions, add options, and resolve choices.

### Action Layer (Workflows)
- `mg.procedure`: Design flows, steps, and affordances.
- `mg.risk`: Assess severity and likelihood for any node.

### Memory Layer (Persistence)
- `mg.sessionOp`: Open, trace, and close conversational sessions.
- `mg.distill`: Synthesize sessions into durable summaries.
- `mg.memoryConfig`: Manage system-wide preferences and policies.

### Agent Layer (Control)
- `mg.plan`: Create tasks and plan execution steps.
- `mg.governance`: Set safety budgets and request approvals.
- `mg.execution`: Register agent runs and track outcomes.

### Connective Tissue
- `mg.retrieve`: Unified search (text, semantic, goals, questions).
- `mg.traverse`: Navigate graph (chains, neighborhood, paths, subgraphs).
- `mg.evolve`: Mutate, tombstone (cascade), and decay salience.

---

## Client API Example

```javascript
const mg = require('./scripts/mindgraph-client.js');

// Example: Evidence-backed reasoning
await mg.addArgument({
  claim: { label: "Project Alpha is ready", content: "All milestones completed." },
  evidence: [{ label: "QA Report", description: "Zero critical bugs." }],
  warrant: { label: "Policy", explanation: "Zero bugs = Launch ready." }
});
```

## References
- [SCHEMA.md](references/SCHEMA.md) — Full node and edge ontology.
- [API.md](references/API.md) — Detailed signatures for the 18 tools.
- [INTEGRATION.md](references/INTEGRATION.md) — Setup guide and opt-in automation.
- [MAINTENANCE.md](references/MAINTENANCE.md) — Heartbeat, dreaming, and watchdog configuration.
