/**
 * MindObserver.js — Output Monitor (JS Side)
 * 
 * Watches the `output/` directory for new autonomous thoughts generated by 
 * the Python Mind Engine. When a valid impulse is detected, it reads the 
 * JSON, emits an event, and then cleans up the file.
 */

const fs = require('fs');
const path = require('path');
const EventEmitter = require('events');

const OUTPUT_DIR = path.resolve(__dirname, '../output');

class MindObserver extends EventEmitter {
    constructor() {
        super();
        this.isWatching = false;
        this.watcher = null;

        // Ensure output dir exists
        if (!fs.existsSync(OUTPUT_DIR)) {
            fs.mkdirSync(OUTPUT_DIR, { recursive: true });
        }
    }

    start() {
        if (this.isWatching) return;
        this.isWatching = true;

        console.log(`[MindObserver] 👁️  Starting to watch: ${OUTPUT_DIR}`);

        // Process any existing files that might have been created while we were offline
        this._scanPendingFiles();

        // Start filesystem watcher
        this.watcher = fs.watch(OUTPUT_DIR, (eventType, filename) => {
            if (eventType === 'rename' && filename && filename.endsWith('.json')) {
                const filePath = path.join(OUTPUT_DIR, filename);
                // The 'rename' event in fs.watch covers both creation and deletion.
                // We only process if the file exists (creation).
                if (fs.existsSync(filePath)) {
                    this._processFile(filePath);
                }
            }
        });
    }

    stop() {
        if (!this.isWatching) return;
        if (this.watcher) {
            this.watcher.close();
            this.watcher = null;
        }
        this.isWatching = false;
        console.log("[MindObserver] 🛑 Stopped watching.");
    }

    _scanPendingFiles() {
        try {
            const files = fs.readdirSync(OUTPUT_DIR);
            for (const file of files) {
                if (file.endsWith('.json')) {
                    this._processFile(path.join(OUTPUT_DIR, file));
                }
            }
        } catch (e) {
            console.error("[MindObserver] Failed to scan pending files:", e);
        }
    }

    _processFile(filePath) {
        // Wait a tiny bit to ensure the Python script has finished writing
        setTimeout(() => {
            if (!fs.existsSync(filePath)) return;

            try {
                const raw = fs.readFileSync(filePath, 'utf8');
                const data = JSON.parse(raw);

                // Emit the autonomous thought event for the Bridge to handle
                this.emit('autonomous_thought', data);

                // Clean up the output file so it's not processed again
                fs.unlinkSync(filePath);

            } catch (e) {
                // If the file is still being written, JSON.parse might fail. 
                // In a production environment, we'd use a more robust locking or .tmp rename pattern on the Python side.
                // Since our Python side does atomic writes, this shouldn't happen often.
                console.warn(`[MindObserver] Error processing ${path.basename(filePath)}: ${e.message}`);
            }
        }, 50);
    }
}

module.exports = MindObserver;
